<?php

namespace framework\pay\wechat;

use framework\Request;
use framework\Exception;
use shangfan\model\base\SystemPayRequest;
use shangfan\model\base\SystemPayNotify;
use shangfan\model\base\Order;

require_once libraryPath() . '/wxpayment/php_sdk_v3.0.10/lib/WxPay.Api.php';
require_once libraryPath() . '/wxpayment/php_sdk_v3.0.10/lib/WxPay.Notify.php';
require_once libraryPath() . '/wxpayment/php_sdk_v3.0.10/lib/WxPay.JsApiPay.php';

/**
 * -----------------------------------------------------------------------------
 * f framework
 * @author df
 * -----------------------------------------------------------------------------
 * 微信支付
 */
class WxPay extends \WxPayNotify {

    /**
     * qrcode
     * @param type $conf
     */
    public function qrcode($conf = array()) {
        $result = $this->unifiedOrder($conf);
        SystemPayRequest::add($conf, $result, 'wechat');
        if ($result && $result['return_code'] == 'SUCCESS' && array_key_exists('code_url', $result)) {
            return base64_encode($result["code_url"]);
        }
        throw new Exception(json_encode($result));
    }

    /**
     * unifiedOrder
     * @param type $conf
     * @return type
     */
    private function unifiedOrder($conf = array()) {
        $input = new \WxPayUnifiedOrder();
        $input->SetBody(array_key_exists('body', $conf) ? $conf['body'] : '');
        $input->SetAttach(array_key_exists('attach', $conf) ? $conf['attach'] : 'attach');
        $input->SetOut_trade_no(array_key_exists('outTradeNo', $conf) ? $conf['outTradeNo'] : '');
        $input->SetTotal_fee(array_key_exists('totalFee', $conf) ? bcmul($conf['totalFee'], 100, 0) : 0);
        $input->SetTime_start(date('YmdHis'));
        $input->SetTime_expire(date('YmdHis', time() + 600));
        $input->SetGoods_tag(array_key_exists('goodsTag', $conf) ? $conf['goodsTag'] : 'WXG');
        $input->SetNotify_url(array_key_exists('notifyUrl', $conf) ? $conf['notifyUrl'] : '');
        $input->SetTrade_type(array_key_exists('tradeType', $conf) ? $conf['tradeType'] : 'NATIVE');
        if (array_key_exists('openid', $conf)) {
            $input->SetOpenid($conf['openid']);
        }
        $input->SetProduct_id(array_key_exists('productId', $conf) ? $conf['productId'] : '0');
        return \WxPayApi::unifiedOrder(new WxPayConfig(), $input);
    }

    /**
     * notify
     * @return type
     * @throws Exception
     */
    public function notify(Request $request) {
        return $this->Handle(new WxPayConfig(), true);
    }

    /**
     * NotifyProcess
     * @param type $objData
     * @param type $config
     * @param type $msg
     */
    public function NotifyProcess($objData, $config, &$msg) {
        if (!count($objData->values)) {
            return false;
        }
        SystemPayNotify::add($objData->values);
        $orderNumberArr = explode(',', $objData->values['out_trade_no']);
        if (!count($orderNumberArr)) {
            return false;
        }
        foreach ($orderNumberArr as $val) {
            $order = Order::detail($val);
            if (!$order || $order->is_pay == 1) {
                continue;
            }
            $order->fill([
                'is_pay' => 1,
                'pay_time' => time(),
                'pay_type' => 2,
                'pay_amount' => $order->total_amount
            ]);
            if (!$order->save()) {
                return false;
            }
        }
        return true;
    }

    /**
     * orderQuery
     * @param $orderNumberStr
     * @return type
     */
    public function orderQuery($orderNumberStr) {
        $wxPayOrderQuery = new \WxPayOrderQuery();
        $wxPayOrderQuery->SetOut_trade_no($orderNumberStr);
        $result = \WxPayApi::orderQuery(new WxPayConfig(), $wxPayOrderQuery);
        if (!$result) {
            return statusFailure();
        }
        if (array_key_exists('trade_state', $result) && $result['trade_state'] == 'SUCCESS') {
            return statusSuccess();
        }
        return statusFailure();
    }

    /**
     * jsApiParameters
     * @return type
     */
    public function jsApiParameters($orderNumberStr) {
        $jsApiPay = new \JsApiPay();
        $openid = $jsApiPay->GetOpenid();
        $orderNumberArr = explode(',', $orderNumberStr);
        $order = $this->unifiedOrder([
            'body' => '订单号' . $orderNumberStr,
            'totalFee' => Order::orderTotalAmount($orderNumberArr),
            'outTradeNo' => $orderNumberStr,
            'notifyUrl' => httpDomain() . '/mall/pay/notify/wechat',
            'tradeType' => 'JSAPI',
            'openid' => $openid
        ]);
        return $jsApiPay->GetJsApiParameters($order);
    }

}
