<?php

declare (strict_types=1);

namespace App\Model\Common;

use Carbon\Carbon;
use Hyperf\Database\Model\SoftDeletes;
use Hyperf\DbConnection\Model\Model;

/**
 * @property int $id
 * @property string $uuid
 * @property int $scene_area_id
 * @property string $name
 * @property int $wxapp_id
 * @property string $cover
 * @property string $description
 * @property string $content
 * @property string $open_start_time
 * @property string $open_end_time
 * @property string $detail_address
 * @property string $longitude
 * @property string $latitude
 * @property int $reading
 * @property int $click
 * @property int $share
 * @property int $collection
 * @property int $oppose
 * @property int $punch
 * @property int $reading_score
 * @property int $share_score
 * @property int $click_score
 * @property int $collection_score
 * @property int $orders
 * @property int $is_show
 * @property int $is_top
 * @property int $is_hot
 * @property Carbon $created_at
 * @property Carbon $updated_at
 * @property string $deleted_at
 * @property string $province_code
 * @property string $city_code
 * @property string $district_code
 * @property string $phone_number
 * @property string $mobile_number
 * @property string $keywords
 * @property int $sale_ticket
 * @property int $star_leave
 */
class SceneSpot extends Model
{
    use SoftDeletes;
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'scene_spot';
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [];
    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'id'               => 'string',
        'scene_area_id'    => 'integer',
        'wxapp_id'         => 'integer',
        'reading'          => 'integer',
        'click'            => 'integer',
        'share'            => 'integer',
        'collection'       => 'integer',
        'oppose'           => 'integer',
        'punch'            => 'integer',
        'reading_score'    => 'integer',
        'share_score'      => 'integer',
        'click_score'      => 'integer',
        'collection_score' => 'integer',
        'orders'           => 'integer',
        'is_show'          => 'integer',
        'is_top'           => 'integer',
        'is_hot'           => 'integer',
        'created_at'       => 'datetime',
        'updated_at'       => 'datetime',
        'sale_ticket'      => 'integer',
        'star_leave'       => 'integer'
    ];

    /**
     * 景点列表
     * @param int $preSize
     * @return array
     */
    public function list(int $preSize): array
    {
        $items = SceneSpot::query()->select([
            'uuid as id',
            'name',
            'open_start_time',
            'open_end_time',
            'star_leave',
            'cover',
        ])->paginate($preSize);

        return [
            'items' => $items->items(),
            'total' => $items->total(),
            'page'  => $items->currentPage(),
            'size'  => $preSize,
        ];
    }

    /**
     * 景点详情
     * @param string $id
     * @return array
     */
    public function detail(string $id): array
    {
        $bean = SceneSpot::query()->where('uuid', '=', $id)->select([
            'uuid as id',
            'name',
            'open_start_time',
            'open_end_time',
            'star_leave',
            'cover',
            'keywords',
            'detail_address',
            'phone_number',
            'mobile_number',
            'longitude',
            'latitude',
            'content',
            'images',
        ])->first();
        if (!empty($bean)) {
            return $bean->toArray();
        } else {
            return [];
        }
    }

    /**
     * 获取器格式化景点标签
     * @param $key
     * @return array
     */
    public function getKeywordsAttribute($key): array
    {
        if (empty($key)) {
            return [];
        } else {
            return explode(',', $key);
        }
    }

    /**
     * 获取器格式化座机号
     * @param $key
     * @return string
     */
    public function getMobileNumberAttribute($key): string
    {
        if (empty($key)) {
            return '';
        }
    }

    /**
     * 获取器格式化手机号
     * @param $key
     * @return string
     */
    public function getPhoneNumberAttribute($key): string
    {
        if (empty($key)) {
            return '';
        }
    }

    /**
     * 格式化景点配图
     * @param $key
     * @return array|mixed
     */
    public function getImagesAttribute($key)
    {
        return empty($key) ? [] : json_decode($key, true);
    }
}