﻿using System;
using System.Data;
using System.Configuration;
using System.Configuration.Provider;
using System.Web.Configuration;
using System.Data.Common;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using Winson.Framework.LogProviders;

namespace Winson.Framework.DBProviders
{
    /// <summary>
    /// 操作方式
    /// </summary>
    public enum DBAction
    {
        Insert,
        Update,
        Delete,
        Other
    }

    public static class DBManage
    {
        private static DataBaseProvider _provider;
        private static DataBaseProviderCollection _providers;
        private static object _lock = new object();


        #region 私有方法

        /// <summary>
        /// 当前Provider
        /// </summary>
        private static DataBaseProvider Provider
        {
            get { GetProviders(); return _provider; }
        }
        /// <summary>
        /// 所有Provider集合
        /// </summary>
        private static DataBaseProviderCollection Providers
        {
            get { GetProviders(); return _providers; }
        }
        /// <summary>
        /// 获取provider
        /// </summary>
        private static void GetProviders()
        {
            //如果 _provider 为空，则创建一个
            if (_provider == null)
            {
                lock (_lock)
                {
                    if (_provider == null)
                    {
                        // 获取 <DBProvider> 配置节
                        DataBaseProviderConfigurationSection section = (DataBaseProviderConfigurationSection)WebConfigurationManager.GetSection("Winson.Framework/DBProvider");

                        _providers = new DataBaseProviderCollection();
                        ProvidersHelper.InstantiateProviders(section.Providers, _providers, typeof(DataBaseProvider));
                        _provider = _providers[section.DefaultProvider];

                        try
                        {
                            if (_provider == null)
                                throw new ProviderException("不能获取默认的 DBProvider");
                        }
                        catch (Exception e)
                        {
                            LogManage.OutputErrLog(e, new Object[] { _provider });
                        }
                    }
                }
            }
        }

        #endregion

        /// <summary>
        /// 获取当前连接对象
        /// </summary>
        public static DbConnection Conn
        {
            get { return Provider.Conn(); }
        }

        #region 公有方法

        /// <summary>
        /// 是否启用事务(默认为启用)
        /// </summary>
        /// <param name="isuse">true:启用  false:不用</param>
        public static void UseTransaction(bool isuse)
        {
            Provider.UseTransaction(isuse);
        }

        /// <summary>
        /// 获取表的主键
        /// </summary>
        /// <param name="tableName">表名</param>
        /// <returns>主键</returns>
        public static string GetPrimeKey(string tableName)
        {
            return Provider.GetPrimeKey(tableName);
        }

        /// <summary>
        /// 生成获取一条实体记录的SQL语句
        /// </summary>
        /// <param name="tableName">表名</param>
        /// <param name="primeKey">关键字</param>
        /// <returns>SQL语句</returns>
        public static string GenerateGetOneEntitySql(string tableName, string primeKey)
        {
            return Provider.GenerateGetOneEntitySql(tableName, primeKey);
        }

        /// <summary>
        /// 生成分页的SQL语句
        /// </summary>
        /// <param name="tableName">表名</param>
        /// <param name="limit">每页显示记录数</param>
        /// <param name="currpage">当前页数</param>
        /// <param name="criteria">查询条件,null为查询所有</param>
        /// <param name="orderby">排序字段</param>
        /// <param name="order">排序方式，默认为 ASC</param>
        /// <returns>SQL语句</returns>
        public static string GeneratePagerSql(string tableName, int limit, int currpage, string criteria, string orderby, string order)
        {
            return Provider.GeneratePagerSql(tableName, limit, currpage, criteria, orderby, order);
        }

        #region ExecuteNonQuery方法
        /// <summary>
        /// 执行单一的SQL语句，没任何参数
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <returns>返回结果记录数</returns>
        public static int ExecuteNonQuery(string strSql)
        {
            return ExecuteNonQuery(DBAction.Other, null, strSql, null, null, null);
        }
        /// <summary>
        /// 执行带条件参数的SQL语句
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <param name="conditionValue">条件参数集合</param>
        /// <returns>返回结果记录数</returns>
        public static int ExecuteNonQuery(string strSql, NameValueCollection conditionValue)
        {
            return ExecuteNonQuery(DBAction.Other, null, strSql, null, null, conditionValue);
        }
        /// <summary>
        /// 无条件参数，根据实体参数执行SQL语句，为Update,Delete,Other等操作
        /// </summary>
        /// <param name="dbAction">执行SQL语句的类型，分别为Update,Delete,Other等操作</param>
        /// <param name="tableName">数据表名</param>
        /// <param name="fieldName">字段名列表</param>
        /// <param name="fieldValue">字段参数值</param>
        /// <returns>返回结果记录数</returns>
        public static int ExecuteNonQuery(DBAction dbAction, string tableName, ArrayList fieldName, ArrayList fieldValue)
        {
            return ExecuteNonQuery(dbAction, tableName, null, fieldName, fieldValue, null);
        }
        /// <summary>
        /// 执行带条件参数并根据实体参数执行SQL语句，分别为Insert,Update,Delete,Other等操作
        /// </summary>
        /// <param name="dbAction">执行SQL语句的类型，分别为Insert,Update,Delete,Other等操作</param>
        /// <param name="tableName">数据表名</param>
        /// <param name="fieldName">字段名列表</param>
        /// <param name="fieldValue">字段参数值</param>
        /// <param name="conditionValue">条件参数值</param>
        /// <returns>返回结果记录数</returns>
        public static int ExecuteNonQuery(DBAction dbAction, string tableName,ArrayList fieldName, ArrayList fieldValue, NameValueCollection conditionValue)
        {
            return ExecuteNonQuery(dbAction, tableName, null, fieldName, fieldValue, conditionValue);
        }
        /// <summary>
        /// 执行SQL语句
        /// </summary>
        /// <param name="dbAction">执行SQL语句的类型，分别为Insert,Update,Delete,Other</param>
        /// <param name="tableName">要操作的表名</param>
        /// <param name="strSql">SQL语句</param>
        /// <param name="fieldName">字段名列表</param>
        /// <param name="fieldValue">字段参数值</param>
        /// <param name="conditionValue">条件参数值</param>
        /// <returns>返回结果记录数</returns>
        public static int ExecuteNonQuery(DBAction dbAction, string tableName, string strSql, ArrayList fieldName, ArrayList fieldValue, NameValueCollection conditionValue)
        {
            return Provider.ExecuteNonQuery(dbAction, tableName, strSql, fieldName, fieldValue, conditionValue);
        }
        #endregion

        #region 批量操作
        /// <summary>
        /// 批量增、删、改操作
        /// </summary>
        /// <param name="dbAction">执行SQL语句的类型，分别为Insert,Update,Delete</param>
        /// <param name="tableName">要操作的表名</param>
        /// <param name="fieldNameList">字段名列表集合</param>
        /// <param name="fieldValueList">字段参数值集合</param>
        /// <returns>成功返回True 否则返回False</returns>
        public static bool ExecuteBatch(DBAction dbAction, string tableName, List<ArrayList> fieldNameList, List<ArrayList> fieldValueList)
        {
            return Provider.ExecuteBatch(dbAction, tableName, fieldNameList, fieldValueList);
        }
        #endregion

        #region ExecuteScalar方法
        /// <summary>
        /// 执行SQL语句，返回第一行第一列的资料
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <returns>返回第一行,第一列的值</returns>
        public static object ExecuteScalar(string sql)
        {
            return ExecuteScalar(sql, null);
        }
        //// <summary>
        /// 执行SQL语句，返回第一行第一列的资料
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <param name="conditionValue">参数集合</param>
        /// <returns>返回第一行,第一列的值</returns>
        public static object ExecuteScalar(string sql, NameValueCollection conditionValue)
        {
            return Provider.ExecuteScalar(sql, conditionValue);
        }
        #endregion

        #region ExecuteReader方法
        /// <summary>
        /// 执行SQL语句，返回DbDataReader
        /// </summary>
        /// <param name="conn">数据库连接对象</param>
        /// <param name="strSql">SQL语句</param>
        /// <returns>返回DbDataReader</returns>
        public static DbDataReader ExecuteReader(DbConnection conn, string strSql)
        {
            return ExecuteReader(conn, strSql, null);
        }
        /// <summary>
        /// 执行SQL语句，返回DbDataReader
        /// </summary>
        /// <param name="conn">数据库连接对象</param>
        /// <param name="strSql">SQL语句</param>
        /// <param name="conditionValue">参数集合</param>
        /// <returns>返回DbDataReader</returns>
        public static DbDataReader ExecuteReader(DbConnection conn, string strSql, NameValueCollection conditionValue)
        {
            return Provider.ExecuteReader(conn, strSql, conditionValue);
        }
        #endregion

        #region ExecuteDataset方法
        /// <summary>
        /// 执行SQL语句，返回DataSet
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <returns>返回DataSet</returns>
        public static DataSet ExecuteDataset(string strSql)
        {
            return ExecuteDataset(strSql, null, null);
        }
        /// <summary>
        /// 执行SQL语句，填充指定表，并返回DataSet
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <param name="tableName">填充的表名</param>
        /// <returns>返回DataSet</returns>
        public static DataSet ExecuteDataset(string strSql, string tableName)
        {
            return ExecuteDataset(strSql, tableName, null);
        }
        /// <summary>
        /// 执行SQL语句，填充指定表，并返回DataSet
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <param name="tableName">填充的表名</param>
        /// <param name="conditionValue">参数集合</param>
        /// <returns>返回DataSet</returns>
        public static DataSet ExecuteDataset(string strSql, string tableName, NameValueCollection conditionValue)
        {
            return Provider.ExecuteDataset(strSql, tableName, conditionValue);
        }
        #endregion 

        #region 执行存储过程
        /// <summary>
        /// 执行存储过程
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <returns>成功返回True 否则返回False</returns>
        public static bool ExecuteProcedure(string procName)
        {
            return ExecuteProcedure(procName, null);
        }
        /// <summary>
        /// 执行存储过程
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <param name="conditionValue">参数列表</param>
        /// <returns>成功返回True 否则返回False</returns>
        public static bool ExecuteProcedure(string procName, NameValueCollection conditionValue)
        {
            return Provider.ExecuteProcedure(procName, conditionValue);
        }
        /// <summary>
        /// 执行存储过程,返回第一行第一列的值 
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <returns>返回第一行,第一列的值</returns>
        public static object ExecuteProcScalar(string procName)
        {
            return ExecuteProcScalar(procName, null);
        }
        /// <summary>
        /// 执行存储过程,返回第一行第一列的值 
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <param name="conditionValue">参数列表</param>
        /// <returns>返回第一行,第一列的值</returns>
        public static object ExecuteProcScalar(string procName, NameValueCollection conditionValue)
        {
            return Provider.ExecuteProcScalar(procName, conditionValue);
        }
        /// <summary>
        /// 执行存储过程，返回DataSet
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <returns>返回DataSet</returns>
        public static DataSet ExecuteProcDataset(string procName)
        {
            return ExecuteProcDataset(procName, null, null);
        }
        /// <summary>
        /// 执行存储过程，返回DataSet
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <param name="tableName">填充表名</param>
        /// <returns>返回DataSet</returns>
        public static DataSet ExecuteProcDataset(string procName, string tableName)
        {
            return ExecuteProcDataset(procName, tableName, null);
        }
        /// <summary>
        /// 执行存储过程，返回DataSet
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <param name="tableName">填充表名</param>
        /// <param name="conditionValue">参数列表</param>
        /// <returns>返回DataSet</returns>
        public static DataSet ExecuteProcDataset(string procName, string tableName, NameValueCollection conditionValue)
        {
            return Provider.ExecuteProcDataset(procName, tableName, conditionValue);
        }
        #endregion

        #endregion
    }
}
