﻿// ===============================================================================
//  解决方案：网鸟Asp.Net模板引擎
//  项目名称：wwwroot
//  文件名称：build.aspx.cs
//  代码作者：网鸟老兵团(Me.YMind)
//  创建时间：2011-06-17 18:26
// ===============================================================================
//  Copyright © Htmlbird.Net. All rights reserved .
//  官方网站：http://www.htmlbird.net/
//  技术论坛：http://bbs.htmlbird.net/
//  版权所有：网鸟IT技术论坛 颜铭工作室
// ===============================================================================

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Web.UI;
using MyWebsite;
using Net.Htmlbird.TemplateEngine;
using Net.Htmlbird.TemplateEngine.Tags;

public partial class Build : Page
{
	private readonly Dictionary<string, string> _fileContents = new Dictionary<string, string>();
	private readonly ApplicationInfo _applicationInfo = new ApplicationInfo(Guid.NewGuid(), "测试产品", "Net.Htmlbird.TemplateEngine For Asp.Net(C#)", new Version(1, 0, 0), DateTime.Now, "http://www.htmlbird.net/", "http://bbs.htmlbird.net/", "颜铭工作室 网鸟IT技术论坛", "YMind Chan");
	private readonly Dictionary<string, Dictionary<FileInfo, PackingTag>> _javascriptFiles = new Dictionary<string, Dictionary<FileInfo, PackingTag>>();
	private readonly Dictionary<string, Dictionary<FileInfo, PackingTag>> _styleSheetFiles = new Dictionary<string, Dictionary<FileInfo, PackingTag>>();
	private AspxTemplateEngine _templateEngine;

	protected override void OnInit(EventArgs e)
	{
		base.OnInit(e);

		var pageName = this.Request.QueryString["t"];

		if (String.IsNullOrWhiteSpace(pageName)) this._GotoIndex();

		// 取得模板文件的完整路径
		var fileName = this.Server.MapPath(String.Format("~/Templates/{0}.aspx", pageName));

		// 检查模板文件是否存在
		if (File.Exists(fileName) == false) this._GotoIndex();

		// 创建模板引擎的实例
		// 每个实例仅处于一个模板
		this._templateEngine = new AspxTemplateEngine(fileName, this.Server.MapPath("~/"), Encoding.UTF8, "mte", true, this._applicationInfo);

		// 订阅js/css打包事件
		this._templateEngine.CreateJavascriptFile += this._TemplateEngineCreateJavascriptFile;
		this._templateEngine.CreateStyleSheetFile += this._TemplateEngineCreateStyleSheetFile;

		// 启用js/css打包功能
		// 这里设置的值，会被模板中的指令设置覆盖
		// 推荐使用模板指令打开此功能
		//_templateEngine.EnableFilePacking = true;

		// 加载多语言引擎
		this._templateEngine.LanguageEngine = new MultiLanguageEngine(this.Server.MapPath("~/Languages"), "zh-CHS");

		// 注册自定义标记
		_templateEngine.RegisterCustomTag<RedirectTag>("redirect");

		// 添加静态变量
		this._templateEngine.StaticVariables.Add("var_book_Name", "MongoDB权威指南");
		this._templateEngine.StaticVariables.Add("var_bbs_Url", "http://www.htmlbird.net");

		// 处置模板并生成aspx脚本
		var aspxContents = this._templateEngine.ToAspx();

		// 保存
		// 如果包含子目录请注意子目录的处理，否则保存时会引发异常
		File.WriteAllText(this.Server.MapPath(String.Format("~/Aspx/{0}.aspx", pageName)), aspxContents.ToString(), Encoding.UTF8);

		// 对打包文件进行后期处理
		this._SaveJavascriptFiles();
		this._SaveStyleSheetFiles();

		/*
		 * 代码打包的示例中省去了压缩算法
		 * 
		 * 代码打包的原理是：
		 *     1、模板引擎收集各个代码文件的路径
		 *     2、这些路径通过事件的方式逐一展现给开发者
		 *     3、开发者按照顺序，甚至再次排序得到的文件列表
		 *     4、开发者将这些文件合并起来
		 *     5、将合并后的内容保存到磁盘文件（file）
		 *     6、将该file的虚拟路径传送回模板引擎
		 *     7、模板引擎剔除已处理的link以及script标记
		 *     8、模板引擎插入新的link和script标记，并将href或src指向刚才传回的虚拟路径
		 *     
		 * CreateJavascriptFile事件和CreateStyleSheetFile事件每次触发都是针对每个模板文档的，其中包含了涉及到的文件列表
		 * 这样做，可以提升效率，同时可以满足个性化定制的需求，因为每个模板都可能会引入不能的脚本和样式表
		 * 
		 * 在本示例中，使用的是文件列表的文件名的哈希值作为最终的虚拟路径的，因为这种实现最简单，o(∩_∩)o 哈哈
		 * 
		 * 关于模板引擎最难以理解的就是这个文件打包功能了，希望大家能够细心体会，它能给你带来很大的帮助！
		 */

		// 完成
		this.statusLabel.Text = "操作完成！";
	}

	private void _TemplateEngineCreateStyleSheetFile(object sender, CreateFileNameEventArgs e)
	{
		if (String.IsNullOrWhiteSpace(this.Context.Request.PhysicalApplicationPath)) return;

		// 取文件列表字符串的 MD5 哈希值作为文件名的唯一标识符
		var hashCode = MD5(String.Join(", ", e.Tags.Select(item => item.Key.FullName).ToArray()));

		// 构建保存路径
		var fileName = this.Server.MapPath(String.Format("Cache/{0}.css", hashCode));

		// 返回代码引用路径给模板引擎，模板引擎将会把这个路径作为 link 标签的 href 属性的值。
		e.FileName = fileName.Replace(this.Context.Request.PhysicalApplicationPath, String.Empty).Replace('\\', '/').Insert(0, "../");

		// 保存到缓存字典，避免对重名文件重复读写造成的性能损失
		if (this._styleSheetFiles.ContainsKey(fileName) == false) this._styleSheetFiles.Add(fileName, e.Tags);
	}

	private void _TemplateEngineCreateJavascriptFile(object sender, CreateFileNameEventArgs e)
	{
		if (String.IsNullOrWhiteSpace(this.Context.Request.PhysicalApplicationPath)) return;

		// 取文件列表字符串的 MD5 哈希值作为文件名的唯一标识符
		var hashCode = MD5(String.Join(", ", e.Tags.Select(item => item.Key.FullName).ToArray()));

		// 构建保存路径
		var fileName = this.Server.MapPath(String.Format("Cache/{0}.js", hashCode));

		// 返回代码引用路径给模板引擎，模板引擎将会把这个路径作为 script 标签的 src 属性的值。
		e.FileName = fileName.Replace(this.Context.Request.PhysicalApplicationPath, String.Empty).Replace('\\', '/').Insert(0, "../");

		// 保存到缓存字典，避免对重名文件重复读写造成的性能损失
		if (this._javascriptFiles.ContainsKey(fileName) == false) this._javascriptFiles.Add(fileName, e.Tags);
	}

	private void _SaveJavascriptFiles()
	{
		foreach (var item in this._javascriptFiles)
		{
			var fileContents = this._GetJavascriptFileContents(item.Value);

			File.WriteAllText(item.Key, fileContents.ToString().Trim(), Encoding.UTF8);
		}
	}

	private void _SaveStyleSheetFiles()
	{
		foreach (var item in this._styleSheetFiles)
		{
			var fileContents = this._GetStyleSheetFileContents(item.Value);

			File.WriteAllText(item.Key, fileContents.ToString().Trim(), Encoding.UTF8);
		}
	}

	private StringBuilder _GetJavascriptFileContents(Dictionary<FileInfo, PackingTag> scripts)
	{
		var codeDocument = new StringBuilder();

		foreach (var pair in scripts)
		{
			using (var reader = pair.Key.OpenText())
			{
				var code = this._fileContents.ContainsKey(pair.Key.FullName) ? this._fileContents[pair.Key.FullName] : reader.ReadToEnd().Trim();

				if (this._fileContents.ContainsKey(pair.Key.FullName) == false) this._fileContents.Add(pair.Key.FullName, code);

				if (String.IsNullOrEmpty(code)) continue;

				if (codeDocument.Length > 0) codeDocument.AppendLine();

				codeDocument.AppendLine(code);
				codeDocument.AppendLine();

				reader.Close();
			}
		}

		return codeDocument;
	}

	private StringBuilder _GetStyleSheetFileContents(Dictionary<FileInfo, PackingTag> styles)
	{
		var codeDocument = new StringBuilder();

		foreach (var pair in styles)
		{
			using (var reader = pair.Key.OpenText())
			{
				var code = this._fileContents.ContainsKey(pair.Key.FullName) ? this._fileContents[pair.Key.FullName] : reader.ReadToEnd().Trim();

				if (this._fileContents.ContainsKey(pair.Key.FullName) == false) this._fileContents.Add(pair.Key.FullName, code);

				if (String.IsNullOrEmpty(code)) continue;

				if (codeDocument.Length > 0)
				{
					var charsetIndex = code.IndexOf("@charset");

					if (charsetIndex >= 0) code = code.Remove(charsetIndex, code.IndexOf(";", charsetIndex + 8) + 1);

					codeDocument.AppendLine();
				}

				codeDocument.AppendLine(code);
				codeDocument.AppendLine();

				reader.Close();
			}
		}

		return codeDocument;
	}

	protected void BtnGoBackClick(object sender, EventArgs e) { this._GotoIndex(); }

	private void _GotoIndex() { this.Response.Redirect("~/"); }

	/// <summary>
	/// 计算指定字符串的 MD5 哈希值，取其中间16位并返回大写形式。
	/// </summary>
	/// <param name="str">指定一个字符串。</param>
	/// <returns>返回指指定字符串的 MD5 哈希值。</returns>
	public static string MD5(string str) { return MD5(str, false); }

	/// <summary>
	/// 计算指定字符串的 MD5 哈希值
	/// </summary>
	/// <param name="str">指定一个字符串。</param>
	/// <param name="notShort">指定一个布尔值，该值指示本方法是否应该返回 MD5 哈希值的短码表示形式。</param>
	/// <returns>返回指指定字符串的 MD5 哈希值。</returns>
	public static string MD5(string str, bool notShort)
	{
		var b = new MD5CryptoServiceProvider().ComputeHash(Encoding.Default.GetBytes(str));

		str = BitConverter.ToString(b).Replace("-", String.Empty).ToUpper();

		if (notShort == false) str = str.Substring(8, 16);

		return str;
	}
}