﻿/* 请提高版权意识，不要删除这些版权声明。
 * --------------------------------------------------------------------------------------
 * 产品名称：问沫聚合门户管理系统
 * 产品作者：Guidy
 * 版权所有：问沫工作室 [VMoom Studio]
 * --------------------------------------------------------------------------------------
 * 官方网站：http://www.vmoom.net/
 * 技术论坛：http://bbs.vmoom.net/
 * ------------------------------------------------------------------------------------*/

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Xml;
using VMoom.Framework.WindowsAPI;

namespace QQLoginManager.Framework
{
	/// <summary>
	/// QQ 登录助手构造类。无从继承此类。
	/// </summary>
	public sealed class QQLoginProvider
	{
		#region 私有字段
		
		private QQAccountCollection _QQAccounts;
		private List<string> _OnLineList;

		#endregion

		#region 构造函数
		
		/// <summary>
		/// 初始化 QQ 登录助手。
		/// </summary>
		public QQLoginProvider()
		{
			_QQAccounts = new QQAccountCollection();
		}

		#endregion

		#region 私有方法

		private bool WindowsProc(IntPtr hWnd, int lParam)
		{
			StringBuilder lpstring = new StringBuilder(0x100);
			Win32API.GetWindowText(hWnd, lpstring, lpstring.Capacity);
			string str = lpstring.ToString();
			string q = "";

			int i = str.IndexOf("_QQMusic_SmallClient", StringComparison.OrdinalIgnoreCase);
			int j = str.IndexOf("QQ.EXE", StringComparison.OrdinalIgnoreCase);

			if (i > 0)
			{
				// ########_QQMusic_SmallClient
				q = str.Remove(i);
			}
			else if (j > -1 && str.Length > j + 42)
			{
				// E:/PROGRAM FILES/TENCENT/QQ/QQ.EXED414641A-643F-4547-8459-D36F7EE9BB0D########
				q = str.Remove(0, j + 42);
			}

			if (q != null && q.Length > 0)
				if (!_OnLineList.Contains(q)) _OnLineList.Add(q);

			return true;
		}

		#endregion

		#region 公有方法

		/// <summary>
		/// 从 QQ 帐号配置文件反序列化以重建登录助手构造类的实例。
		/// </summary>
		/// <param name="filename">包含配置文件完整路径的字符串。</param>
		/// <returns>QQLoginManager.Framework.QQLoginProvider</returns>
		public static QQLoginProvider FromFile(string filename)
		{
			QQLoginProvider c = new QQLoginProvider();

			XmlDocument doc = new XmlDocument();

			if (File.Exists(filename) == true)
			{
				doc.Load(filename);

				XmlNodeList xnl = doc.SelectNodes("/QQLoginManager/QQAccounts/QQ");

				foreach (XmlNode node in xnl)
					c._QQAccounts.Add(node.Attributes["UserName"].Value, node.Attributes["UserPassHash"].Value, node.Attributes["State"].Value);
			}

			return c;
		}

		/// <summary>
		/// 将 QQ 帐号配置信息保存到文件。
		/// </summary>
		/// <param name="filename"></param>
		public void SaveToFile(string filename)
		{
			XmlDocument doc = new XmlDocument();
			XmlNode node;
			XmlElement element;

			node = doc.CreateNode(XmlNodeType.XmlDeclaration, "", "");
			doc.AppendChild(node);

			element = doc.CreateElement("", "QQLoginManager", "");
			element.AppendChild(doc.CreateElement("QQAccounts"));
			doc.AppendChild(element);

			foreach (QQAccount qq in _QQAccounts)
			{
				element = doc.CreateElement("QQ");

				element.SetAttribute("UserName", qq.UserName);
				element.SetAttribute("UserPassHash", qq.UserPassHash);
				element.SetAttribute("State", ((int)qq.InitState).ToString());

				node = doc.SelectSingleNode("/QQLoginManager/QQAccounts");
				node.AppendChild(element);
			}

			doc.Save(filename);
		}

		/// <summary>
		/// 添加 QQ 帐号。
		/// </summary>
		/// <param name="username">包含 QQ 帐号名称的字符串。</param>
		/// <param name="userpass">包含 QQ 帐号密码的字符串。</param>
		/// <param name="state">包含 QQ 登录状态的枚举。</param>
		public void AddQQ(string username, string userpass, QQLoginState state)
		{
			_QQAccounts.Add(username, userpass, state);
		}

		/// <summary>
		/// 删除 QQ 帐号。
		/// </summary>
		/// <param name="username">包含 QQ 帐号名称的字符串。</param>
		public void RemoveQQ(string username)
		{
			_QQAccounts.Remove(username);
		}

		/// <summary>
		/// 刷新 QQ 登录状态。
		/// </summary>
		public void Refresh()
		{
			if (_OnLineList == null) _OnLineList = new List<string>();
			_OnLineList.Clear();

			Win32API.EnumWindowsProc x = new Win32API.EnumWindowsProc(this.WindowsProc);
			Win32API.EnumWindows(x, 0);

			foreach (QQAccount qq in _QQAccounts)
				qq.CurrState = this.CheckState(qq);
		}

		/// <summary>
		/// 登录所有 QQ 。
		/// </summary>
		public void Login(string qqpath)
		{
			this.Refresh();

			foreach (QQAccount qq in _QQAccounts)
				if (qq.CurrState == QQLoginState.Offline) qq.Login(qqpath);
		}

		/// <summary>
		/// 获取指定 QQ 号码的登录状态。
		/// </summary>
		/// <param name="qq">QQ 帐号实例。</param>
		/// <returns>QQLoginState</returns>
		public QQLoginState CheckState(QQAccount qq)
		{
			return this.CheckState(qq.UserName);
		}

		/// <summary>
		/// 获取指定 QQ 号码的登录状态。
		/// </summary>
		/// <param name="username">指定一个包含 QQ 帐号信息的字符串。</param>
		/// <returns>QQLoginState</returns>
		public QQLoginState CheckState(string username)
		{
			return (_OnLineList.Contains(username) == true) ? QQLoginState.Online : QQLoginState.Offline;
		}

		#endregion

		#region 公有属性
		
		/// <summary>
		/// 获取 QQ 帐号信息集合。
		/// </summary>
		public QQAccountCollection QQAccounts
		{
			get { return _QQAccounts; }
		}

		#endregion
	}
}
