﻿/* 请提高版权意识，不要删除这些版权声明。
 * --------------------------------------------------------------------------------------
 * 产品名称：问沫聚合门户管理系统
 * 产品作者：Guidy
 * 版权所有：问沫工作室 [VMoom Studio]
 * --------------------------------------------------------------------------------------
 * 官方网站：http://www.vmoom.net/
 * 技术论坛：http://bbs.vmoom.net/
 * ------------------------------------------------------------------------------------*/

using System;
using System.Text;

namespace QQLoginManager.Framework
{
	/// <summary>
	/// 标准 XXTEA 加密算法的实现。本算法支持自定义密钥，且为可逆的。
	/// </summary>
	public class XXTEA
	{
		#region 私有字段

		private string _Key = "";
		
		#endregion

		#region 构造函数

		/// <summary>
		/// 初始化 XXTEA 算法。
		/// </summary>
		public XXTEA()
		{
			this.Key = "Key";
		}

		/// <summary>
		/// 使用密钥初始化 XXTEA 算法。
		/// </summary>
		/// <param name="key">要使用的密钥。</param>
		public XXTEA(string key)
		{
			this.Key = key;
		}

		#endregion

		#region 算法实现

		private Byte[] _Encrypt(Byte[] Data, Byte[] Key)
		{
			if (Data.Length == 0)
			{
				return Data;
			}

			return _ToByteArray(_Encrypt(_ToUInt32Array(Data, true), _ToUInt32Array(Key, false)), false);
		}

		private Byte[] _Decrypt(Byte[] Data, Byte[] Key)
		{
			if (Data.Length == 0)
			{
				return Data;
			}

			return _ToByteArray(_Decrypt(_ToUInt32Array(Data, false), _ToUInt32Array(Key, false)), true);
		}

		private UInt32[] _Encrypt(UInt32[] v, UInt32[] k)
		{
			Int32 n = v.Length - 1;
			if (n < 1)
			{
				return v;
			}

			if (k.Length < 4)
			{
				UInt32[] Key = new UInt32[4];
				k.CopyTo(Key, 0);
				k = Key;
			}

			UInt32 z = v[n], y = v[0], delta = 0x9E3779B9, sum = 0, e;
			Int32 p, q = 6 + 52 / (n + 1);

			while (q-- > 0)
			{
				sum = unchecked(sum + delta);
				e = sum >> 2 & 3;
				for (p = 0; p < n; p++)
				{
					y = v[p + 1];
					z = unchecked(v[p] += (z >> 5 ^ y << 2) + (y >> 3 ^ z << 4) ^ (sum ^ y) + (k[p & 3 ^ e] ^ z));
				}
				y = v[0];
				z = unchecked(v[n] += (z >> 5 ^ y << 2) + (y >> 3 ^ z << 4) ^ (sum ^ y) + (k[p & 3 ^ e] ^ z));
			}

			return v;
		}

		private UInt32[] _Decrypt(UInt32[] v, UInt32[] k)
		{
			Int32 n = v.Length - 1;

			if (n < 1)
			{
				return v;
			}

			if (k.Length < 4)
			{
				UInt32[] Key = new UInt32[4];
				k.CopyTo(Key, 0);
				k = Key;
			}

			UInt32 z = v[n], y = v[0], delta = 0x9E3779B9, sum, e;
			Int32 p, q = 6 + 52 / (n + 1);
			sum = unchecked((UInt32)(q * delta));

			while (sum != 0)
			{
				e = sum >> 2 & 3;
				for (p = n; p > 0; p--)
				{
					z = v[p - 1];
					y = unchecked(v[p] -= (z >> 5 ^ y << 2) + (y >> 3 ^ z << 4) ^ (sum ^ y) + (k[p & 3 ^ e] ^ z));
				}
				z = v[n];
				y = unchecked(v[0] -= (z >> 5 ^ y << 2) + (y >> 3 ^ z << 4) ^ (sum ^ y) + (k[p & 3 ^ e] ^ z));
				sum = unchecked(sum - delta);
			}

			return v;
		}

		private UInt32[] _ToUInt32Array(Byte[] Data, Boolean IncludeLength)
		{
			Int32 n = (((Data.Length & 3) == 0) ? (Data.Length >> 2) : ((Data.Length >> 2) + 1));
			UInt32[] Result;

			if (IncludeLength)
			{
				Result = new UInt32[n + 1];
				Result[n] = (UInt32)Data.Length;
			}
			else
			{
				Result = new UInt32[n];
			}

			n = Data.Length;
			for (Int32 i = 0; i < n; i++)
			{
				Result[i >> 2] |= (UInt32)Data[i] << ((i & 3) << 3);
			}

			return Result;
		}

		private Byte[] _ToByteArray(UInt32[] Data, Boolean IncludeLength)
		{
			Int32 n;

			if (IncludeLength)
			{
				n = (Int32)Data[Data.Length - 1];
			}
			else
			{
				n = Data.Length << 2;
			}

			Byte[] Result = new Byte[n];
			for (Int32 i = 0; i < n; i++)
			{
				Result[i] = (Byte)(Data[i >> 2] >> ((i & 3) << 3));
			}

			return Result;
		}

		#endregion

		#region 使用方法

		/// <summary>
		/// 加密字符串
		/// </summary>
		/// <param name="s">传入的明文</param>
		/// <returns>返回密文</returns>
		public string Encrypt(string s)
		{
			Encoding e = Encoding.UTF8;
			Byte[] b = _Encrypt(e.GetBytes(s), e.GetBytes(this.Key));
			return Convert.ToBase64String(b);
		}

		/// <summary>
		/// 解密字符串
		/// </summary>
		/// <param name="s">传入的加密字符串</param>
		/// <returns>返回明文</returns>
		public string Decrypt(string s)
		{
			Encoding e = Encoding.UTF8;

			string ss = string.Empty;

			try
			{
				ss = e.GetString(_Decrypt(Convert.FromBase64String(s), e.GetBytes("Key")));
			}
			catch (Exception)
			{
				return null;
			}

			return ss;
		}

		#endregion

		#region 公有属性

		/// <summary>
		/// 获取密钥（密钥仅对对称算法有效）。
		/// </summary>
		public string Key
		{
			get { return _Key; }
			set { _Key = value; }
		}

		#endregion
	}
}
