import os
import sys


def isWindowsPlatform():
    """
    Function to check, if this is a Windows platform.

    @return flag indicating Windows platform (boolean)
    """
    return sys.platform.startswith(("win", "cygwin"))


def isMacPlatform():
    """
    Function to check, if this is a Mac platform.

    @return flag indicating Mac platform (boolean)
    """
    return sys.platform == "darwin"


def isLinuxPlatform():
    """
    Function to check, if this is a Linux platform.

    @return flag indicating Linux platform (boolean)
    """
    return sys.platform.startswith("linux")


def desktopName():
    """
    Function to determine the name of the desktop environment used
    (Linux only).

    @return name of the desktop environment
    @rtype str
    """
    if not isLinuxPlatform():
        return ""

    currDesktop = os.environ.get("XDG_CURRENT_DESKTOP", "")
    if currDesktop:
        return currDesktop

    currDesktop = os.environ.get("XDG_SESSION_DESKTOP", "")
    if currDesktop:
        return currDesktop

    currDesktop = os.environ.get("GDMSESSION", "")
    if currDesktop:
        return currDesktop

    currDesktop = os.environ.get("GNOME_DESKTOP_SESSION_ID", "")
    if currDesktop:
        return currDesktop

    currDesktop = os.environ.get("KDE_FULL_SESSION", "")
    if currDesktop:
        return currDesktop

    currDesktop = os.environ.get("DESKTOP_SESSION", "")
    if currDesktop:
        return currDesktop

    return ""


def isKdeDesktop():
    """
    Function to check, if the current session is a KDE desktop (Linux only).

    @return flag indicating a KDE desktop
    @rtype bool
    """
    if not isLinuxPlatform():
        return False

    isKDE = False

    desktop = (
            os.environ.get("XDG_CURRENT_DESKTOP", "").lower() or
            os.environ.get("XDG_SESSION_DESKTOP", "").lower() or
            os.environ.get("DESKTOP_SESSION", "").lower()
    )
    if desktop:
        isKDE = "kde" in desktop or "plasma" in desktop
    else:
        isKDE = bool(os.environ.get("KDE_FULL_SESSION", ""))

    return isKDE


def isGnomeDesktop():
    """
    Function to check, if the current session is a Gnome desktop (Linux only).

    @return flag indicating a Gnome desktop
    @rtype bool
    """
    if not isLinuxPlatform():
        return False

    isGnome = False

    desktop = (
            os.environ.get("XDG_CURRENT_DESKTOP", "").lower() or
            os.environ.get("XDG_SESSION_DESKTOP", "").lower() or
            os.environ.get("GDMSESSION", "").lower()
    )
    if desktop:
        isGnome = "gnome" in desktop
    else:
        isGnome = bool(os.environ.get("GNOME_DESKTOP_SESSION_ID", ""))

    return isGnome


def sessionType():
    """
    Function to determine the name of the running session (Linux only).

    @return name of the desktop environment
    @rtype str
    """
    if not isLinuxPlatform():
        return ""

    sessionType = os.environ.get("XDG_SESSION_TYPE").lower()
    if "x11" in sessionType:
        return "X11"
    elif "wayland" in sessionType:
        return "Wayland"

    sessionType = os.environ.get("WAYLAND_DISPLAY", "").lower()
    if "wayland" in sessionType:
        return "Wayland"

    return ""


def isWaylandSession():
    """
    Function to check, if the current session is a wayland session.

    @return flag indicating a wayland session
    @rtype bool
    """
    return sessionType() == "Wayland"
