﻿/**
 * Constructor for LabeledMarker, which picks up on strings from the GMarker
 * options array, and then calls the GMarker constructor.
 *
 * @param {GLatLng} latlng
 * @param {GMarkerOptions} Named optional arguments:
 *   opt_opts.labelText {String} text to place in the overlay div.
 *   opt_opts.labelClass {String} class to use for the overlay div.
 *     (default "LabeledMarker_markerLabel")
 *   opt_opts.labelOffset {GSize} label offset, the x- and y-distance between
 *     the marker's latlng and the upper-left corner of the text div.
 */
function LabeledMarker(latlng, opt_opts){
  this.latlng_ = latlng;
  this.opts_ = opt_opts;

  this.labelText_ = opt_opts.labelText || "";
  this.labelClass_ = opt_opts.labelClass || "LabeledMarker_markerLabel";
  this.labelOffset_ = opt_opts.labelOffset || new GSize(0, 0);

  this.clickable_ = ('undefined' != typeof opt_opts.clickable) ? opt_opts.clickable : true;
  //this.clickable_ = opt_opts.clickable || true;
  this.title_ = opt_opts.title || "";
  this.labelVisibility_  = true;

  if (opt_opts.draggable) {
  	// This version of LabeledMarker doesn't support dragging.
  	opt_opts.draggable = false;
  }
  GMarker.apply(this, arguments);
  
}


// It's a limitation of JavaScript inheritance that we can't conveniently
// inherit from GMarker without having to run its constructor. In order for
// the constructor to run, it requires some dummy GLatLng.
LabeledMarker.prototype = new BMap.Marker();

/**
 * Is called by GMap2's addOverlay method. Creates the text div and adds it
 * to the relevant parent div.
 *
 * @param {GMap2} map the map that has had this labeledmarker added to it.
 */
LabeledMarker.prototype.initialize = function(map) {
  // Do the GMarker constructor first.
  //GMarker.prototype.initialize.apply(this, arguments);
   this.map_ = map;
  this.div_ = document.createElement("div");
  this.div_.className = this.labelClass_;
  this.div_.innerHTML = this.labelText_;
  this.div_.style.position = "absolute";
  this.div_.style.cursor = "pointer";
  this.div_.title = this.title_;

  map.getPane(G_MAP_MARKER_PANE).appendChild(this.div_);

  if (this.clickable_) {
    /**
     * Creates a closure for passing events through to the source marker
     * This is located in here to avoid cluttering the global namespace.
     * The downside is that the local variables from initialize() continue
     * to occupy space on the stack.
     *
     * @param {Object} object to receive event trigger.
     * @param {GEventListener} event to be triggered.
     */
    function newEventPassthru(obj, event) {
      return function() {
        GEvent.trigger(obj, event);
      };
    }

    // Pass through events fired on the text div to the marker.
    var eventPassthrus = ['click', 'dblclick', 'mousedown', 'mouseup', 'mouseover', 'mouseout'];
    for(var i = 0; i < eventPassthrus.length; i++) {
      var name = eventPassthrus[i];
      GEvent.addDomListener(this.div_, name, newEventPassthru(this, name));
    }
  }
};

/**
 * Call the redraw() handler in GMarker and our our redrawLabel() function.
 *
 * @param {Boolean} force will be true when pixel coordinates need to be recomputed.
 */
LabeledMarker.prototype.redraw = function(force) {
  GMarker.prototype.redraw.apply(this, arguments);
  this.redrawLabel_();
};

/**
 * Moves the text div based on current projection and zoom level.
 */
LabeledMarker.prototype.redrawLabel_ = function() {
  // Calculate the DIV coordinates of two opposite corners of our bounds to
  // get the size and position of our rectangle
  var p = this.map_.fromLatLngToDivPixel(this.latlng_);
  var z = GOverlay.getZIndex(this.latlng_.lat());

  // Now position our div based on the div coordinates of our bounds
  this.div_.style.left = (p.x + this.labelOffset_.width) + "px";
  this.div_.style.top = (p.y + this.labelOffset_.height) + "px";
  this.div_.style.zIndex = z; // in front of the marker
};

/**
 * Remove the text div from the map pane, destroy event passthrus, and calls the
 * default remove() handler in GMarker.
 */
 LabeledMarker.prototype.remove = function() {
  GEvent.clearInstanceListeners(this.div_);
  if (this.div_.outerHTML) {
    this.div_.outerHTML = ""; //prevent pseudo-leak in IE
  }
  if (this.div_.parentNode) {
    this.div_.parentNode.removeChild(this.div_);
  }
  this.div_ = null;
  GMarker.prototype.remove.apply(this, arguments);
};

/**
 * Return a copy of this overlay, for the parent Map to duplicate itself in full. This
 * is part of the Overlay interface and is used, for example, to copy everything in the
 * main view into the mini-map.
 */
LabeledMarker.prototype.copy = function() {
  return new LabeledMarker(this.latlng_, this.opts_);
};


/**
 * Shows the marker, and shows label if it wasn't hidden. Note that this function
 * triggers the event GMarker.visibilitychanged in case the marker is currently hidden.
 */
LabeledMarker.prototype.show = function() {
  GMarker.prototype.show.apply(this, arguments);
  if (this.labelVisibility_) {
    this.showLabel();
  } else {
    this.hideLabel();
  }
};


/**
 * Hides the marker and label if it is currently visible. Note that this function
 * triggers the event GMarker.visibilitychanged in case the marker is currently visible.
 */
LabeledMarker.prototype.hide = function() {
  GMarker.prototype.hide.apply(this, arguments);
  this.hideLabel();
};


/**
 * Repositions label and marker when setLatLng is called.
 */
LabeledMarker.prototype.setLatLng = function(latlng) {
  this.latlng_ = latlng;
  GMarker.prototype.setLatLng.apply(this, arguments);
  this.redrawLabel_();
};

/**
 * Sets the visibility of the label, which will be respected during show/hides.
 * If marker is visible when set, it will show or hide label appropriately.
 */
LabeledMarker.prototype.setLabelVisibility = function(visibility) {
  this.labelVisibility_ = visibility;
  if (!this.isHidden()) { // Marker showing, make visible change
    if (this.labelVisibility_) {
      this.showLabel();
    } else {
      this.hideLabel();
    }
  }
};


/**
 * Returns whether label visibility is set on.
 * @return {Boolean}
 */
LabeledMarker.prototype.getLabelVisibility = function() {
  return this.labelVisibility_;
};


/**
 * Hides the label of the marker.
 */
LabeledMarker.prototype.hideLabel = function() {
  this.div_.style.visibility = 'hidden';
};


/**
 * Shows the label of the marker.
 */
LabeledMarker.prototype.showLabel = function() {
  this.div_.style.visibility = 'visible';
};


/**
* 驾车标识起终点时用
*/
var Label = function(style) {
  this.style_ = style ? style : 'border:1px solid #666666;padding:2px;font-size:1em;color:#333333;background:#ffffff;';
};
Label.prototype = new GOverlay();
Label.prototype.initialize = function (map) {
  this.map_ = map;
  this.container_ = document.createElement('div');
  var style = this.style_ + 'position:absolute;white-space:nowrap;';
  this.container_.style.cssText = style;
  SFMap.map._map.getPane(G_MAP_MARKER_PANE).appendChild(this.container_);
};
Label.prototype.setLabel = function (text) {
  this.label_ = text;
  this.redraw();
};
Label.prototype.redraw = function () {
  if (this.latlng_ && this.label_) {
    this.container_.innerHTML = this.label_;
    var point = this.map_.fromLatLngToDivPixel(this.latlng_);
    this.container_.style.left = point.x + 'px';
    this.container_.style.top = point.y + 'px';
  }
};
Label.prototype.show = function(latlng, text) {
  if (latlng) {
    this.latlng_ = latlng;
  }
  if (text) {
    this.label_ = text;
  }
  this.container_.style.display = '';
  this.redraw();
};
Label.prototype.hide = function () {
  this.container_.style.display = 'none';
};
Label.prototype.remove = function () {
  this.container_.parentNode.removeChild(this.container_);
};



// 复杂的自定义覆盖物
    function ComplexCustomOverlay(latlng,obj,provalue, opt_opts){
       this.latlng_ = latlng;
	  this.opts_ = opt_opts;
	  this.obj_=obj;
	  this.provalue_=provalue;
	  this.labelText_ = opt_opts.labelText || "";
	  this.labelClass_ = opt_opts.labelClass || "LabeledMarker_markerLabel";
	  this.labelOffset_ = opt_opts.labelOffset || new GSize(0, 0);
	  this.pointer_=opt_opts.pointer;
	  this.clickable_ = ('undefined' != typeof opt_opts.clickable) ? opt_opts.clickable : true;
	  //this.clickable_ = opt_opts.clickable || true;
	  this.title_ = opt_opts.title || "";
	  this.labelVisibility_  = true;
	
	  if (opt_opts.draggable) {
		// This version of LabeledMarker doesn't support dragging.
		opt_opts.draggable = false;
	  }
    }
    ComplexCustomOverlay.prototype =new BMap.Overlay();
    ComplexCustomOverlay.prototype.initialize = function(map){
      this._map = map;
      var div = this.div_ = document.createElement("div");
	  this.div_.className = this.labelClass_;
	  this.div_.innerHTML = this.labelText_;
	  this.div_.style.position = "absolute";
	  this.div_.style.cursor = "pointer";
	  this.div_.title = this.title_;
	  this.div_.id="tip"+this.provalue_.newCode;
	  this.div_.style.zIndex = BMap.Overlay.getZIndex(this.pointer_.lat);
      var that = this;
      var _provalue = this.provalue_;
	  var _obj=this.obj_;
      div.onmouseover = function(){
          _obj.hoverMarker(that,_provalue,true);
      }
	  div.onmouseout = function(){
          _obj.hoverMarker(that,_provalue,false);
      }
      div.onclick = function(){
		  _obj.openTip(that,_provalue);
      }

      map.getPanes().labelPane.appendChild(div);
      return div;
    }
    ComplexCustomOverlay.prototype.draw = function(){
          var map = this._map;
		  var pixel = map.pointToOverlayPixel(new BMap.Point(this.pointer_.lat,this.pointer_.lng));
		  this.div_.style.left = pixel.x +  "px";
		  this.div_.style.top  = pixel.y - 30 + "px";

    }