/*
 * ArticleIndex.java
 *
 * Created on 2006126, 5:09
 *
 * To change this template, choose Tools | Options and locate the template under
 * the Source Creation and Management node. Right-click the template and choose
 * Open. You can then make changes to the template in the Source Editor.
 */

package tot.search;
import tot.global.Sysconfig;
import tot.bean.DataField;
import tot.util.*;
import tot.exception.SearchException;
import java.io.IOException;
import java.sql.Timestamp;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.*;
import org.apache.lucene.document.DateTools.Resolution;
import org.apache.lucene.index.*;
import org.apache.lucene.store.Directory;
/**
 *
 * @author Administrator
 */
public class ArticleIndex {
    private static Log log = LogFactory.getLog(ArticleIndex.class);
    /** Creates a new instance of ArticleIndex */
    private static Analyzer analyzer;
    private static String searchArticleIndexDir;    
    private static long lastOptimizeTime = 0;    
    static {
        searchArticleIndexDir = Sysconfig.getIndexDir();
        initializeAnalyzer();
    }    
    public static void scheduleAddTask(DataField df) {
        AddUpdateIndexTask task =new AddUpdateIndexTask(df,AddUpdateIndexTask.OPERATION_ADD);
        TimerUtil.getInstance().schedule(task, 0);
    }
    
    public static void scheduleUpdateTask(DataField df) {
        AddUpdateIndexTask task = new AddUpdateIndexTask(df, AddUpdateIndexTask.OPERATION_UPDATE);
        TimerUtil.getInstance().schedule(task, 0);
    }
    public static void scheduleDeleteTask(int objectID) {
        DeleteIndexTask task = new DeleteIndexTask(objectID);
        TimerUtil.getInstance().schedule(task, 0);
    }
    public static void scheduleRebuildIndexTask() {
        int maxPostID = 0;
        RebuildIndexTask task = new RebuildIndexTask(maxPostID);
        TimerUtil.getInstance().schedule(task, 0);
    }
    static Analyzer getAnalyzer() {
        return analyzer;
    }
    
    /**
     * This class will load analyzer when starting. If specified analyzer class
     * cannot be loaded then default analyzer will be used.
     */
    private static void initializeAnalyzer() {
        String analyzerClassName = Sysconfig.getLuceneAnalyzerClassName();
        if ( (analyzerClassName == null) || (analyzerClassName.equals("")) ) {
            //create standard analyzer
            //String[] stopWords = this.loadStopWords();
            analyzer = new StandardAnalyzer();
            log.debug("Using StandardAnalyzer for indexing");
        } else {
            //try to create specified analyzer
            try {
                log.debug("About to load Analyzer [" + analyzerClassName + "] for indexing");
                analyzer = (Analyzer) Class.forName(analyzerClassName).newInstance();
            } catch (Exception e) {
                log.warn("Cannot load " + analyzerClassName + ". Loading StandardAnalyzer");
                analyzer = new StandardAnalyzer();
            }
        }
    }
    
    /**
     * This method is used for getting new IndexWriter. It can create new index
     * or add df to existing index. Creating new index will delete previous so it
     * should be used for rebuilding index.
     * @param create - true if new index should be created.
     *               - false for adding dfs to existing index
     * @return IndexWriter object that is used for adding dfs to index
     */
    static IndexWriter getIndexWriter(Directory directory, boolean create) {
        IndexWriter writer = null;
        //If create = false, we will create IndexWriter with false argument        
        try {
            writer = new IndexWriter(directory, analyzer, create);
            writer.setUseCompoundFile(true);            
        } catch (IOException e) {
            log.warn("Cannot open existed index. New index will be created.", e);
            //Ignore Exception. We will try to create index with true parameter
        }
        return writer;
    }
    
    /**
     * This method is used for adding single df to index
     * Note: this method doesnt close the writer
     * @param df A df that should be indexed
     * @param writer IndexWriter that is used for storing
     * @throws SearchException
     */
    static void doIndex(DataField df, IndexWriter writer) throws SearchException {
        if (df == null) return;
        String title=df.getFieldValue("Title");
        String content=df.getFieldValue("Content");
        String Summary=df.getFieldValue("Summary");
        String datetime=df.getFieldValue("ModiTime");
        // must include topic and body. If not then we have nothing to index.
        if ( (title == null || title.equals("")) ||
                (content == null || content.equals(""))) {
            return;
        }
        if(Summary==null) Summary="...";
        //Each df will be represented as a document
        Document dfDocument = new Document();
        //Document has following fields that could be queried on       
        
        dfDocument.add(new Field("id", df.getFieldValue("id"),Field.Store.YES,Field.Index.UN_TOKENIZED));        
        dfDocument.add(new Field("Hits", df.getFieldValue("Hits"),Field.Store.YES,Field.Index.UN_TOKENIZED));
        //dfDocument.add(Field.Keyword("UserName", df.getFieldValue("UserName")));
        dfDocument.add(new Field("CatalogId", df.getFieldValue("CategoryId"),Field.Store.YES,Field.Index.UN_TOKENIZED));
        //document fileurl,body and title is not stored since we can retrieve them from database
        dfDocument.add(new Field("SavePath", df.getFieldValue("SavePath"),Field.Store.YES,Field.Index.UN_TOKENIZED));
        dfDocument.add(new Field("Title", title, Field.Store.YES, Field.Index.TOKENIZED));
        dfDocument.add(new Field("Summary", Summary, Field.Store.YES, Field.Index.TOKENIZED));
        dfDocument.add(new Field("Content", df.getFieldValue("Content"), Field.Store.NO, Field.Index.TOKENIZED));
        //add date field
        dfDocument.add(new Field("ModiTime", DateTools.dateToString(Timestamp.valueOf(datetime),Resolution.MILLISECOND), Field.Store.YES, Field.Index.UN_TOKENIZED));
        
        //now we have created document with fields so we can store it
        try {
            writer.addDocument(dfDocument);
        } catch (IOException e) {
            log.error("ArticleIndexer.doIndex failed", e);
            //@todo : localize me
            throw new SearchException("Error writing new df to index");
        }
    }
    
    /**
     * Add single df to index
     * @param df
     * @throws SearchException
     */
    static void addToIndex(DataField df) throws SearchException, IOException {
        Directory directory = null;
        IndexWriter writer = null;
        try {
            directory = SearchService.getSearchIndexDir();
            writer = getIndexWriter(directory,false);
            if (writer == null) {
                log.warn("Cannot get the IndexWriter");
                return;
            }
            doIndex(df, writer);
            
            // now check if we should optimize index (each hour)
            long now = System.currentTimeMillis();
            long timeFromLastOptimize = now - lastOptimizeTime;
            if (timeFromLastOptimize > DateUtil.HOUR) {
                log.debug("writer.optimize() called in addToIndex");
                writer.optimize();
                lastOptimizeTime = now;
            }
        } catch (SearchException ex) {
            throw ex;
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (IOException e) {
                    log.debug("Error closing Lucene IndexWriter", e);
                }
            }
            if (directory != null) {
                try {
                    directory.close();
                } catch (IOException e) {
                    log.debug("Cannot close directory.", e);
                }
            }
        }
    }
    
    /**
     * This method is used for deleting df from index.
     * @param dfID id of the df that should be deleted
     * @throws SearchException
     */
    static void deleteFromIndex(int dfID) throws SearchException {
        Directory directory = null;
        IndexReader reader = null;
        try {
            directory = SearchService.getSearchIndexDir();
            reader = IndexReader.open(directory);
            if (reader == null) {
                log.warn("Cannot get the IndexReader");
                return;
            }
            Term term = new Term("id", String.valueOf(dfID));
            int deletedCount = reader.deleteDocuments(term);
            log.debug("deleteFromIndex: deleted articles = " + deletedCount);
        } catch (IOException e) {
            //@todo : localize me
            throw new SearchException("Error trying to delete df with article id = " + dfID);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    log.debug("Error closing Lucene IndexReader", e);
                }
            }
            if (directory != null) {
                try {
                    directory.close();
                } catch (IOException e) {
                    log.debug("Cannot close directory.", e);
                }
            }
        }
    }    
        
    public static int getNumDocs() {
        int numDocs = -1;
        Directory directory = null;
        IndexReader reader = null;
        try {
            directory = SearchService.getSearchIndexDir();
            reader = IndexReader.open(directory);
            if (reader == null) {
                log.warn("Cannot get the IndexReader");
                return -1;
            }
            numDocs = reader.numDocs();
        } catch ( IOException ioe) {
            //ignore
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    log.debug("Error closing Lucene IndexReader", e);
                }
            }
            if (directory != null) {
                try {
                    directory.close();
                } catch (IOException e) {
                    log.debug("Cannot close directory.", e);
                }
            }
        }
        return numDocs;
    }
    
}
