/*
 * SearchQuery.java
 *
 * Created on 2006127, 2:28
 *
 * To change this template, choose Tools | Options and locate the template under
 * the Source Creation and Management node. Right-click the template and choose
 * Open. You can then make changes to the template in the Source Editor.
 */
package tot.search;

import java.io.IOException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import tot.global.Sysconfig;
import tot.dao.DaoFactory;
import tot.bean.DataField;
import tot.util.*;
import tot.search.filter.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.DateTools;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.DateTools.Resolution;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.*;
import org.apache.lucene.store.Directory;

/**
 *
 * @author Administrator
 */
public class SearchQuery {

    private static Log log = LogFactory.getLog(SearchQuery.class);
    // constant for search by time
    public static final int SEARCH_ANY_DATE = 0;
    public static final int SEARCH_NEWER = 1;
    public static final int SEARCH_OLDER = 2;
    // constant for search by the blog title/body
    public static final int SEARCH_ONLY_TITLE = 1;
    public static final int SEARCH_ONLY_BODY = 2;
    // constant for sorting option by time
    public static final int SEARCH_SORT_DEFAULT = 0;
    public static final int SEARCH_SORT_TIME_DESC = 1;
    public static final int SEARCH_SORT_TIME_ASC = 2;
    private String userName = null;
    private int catalogid = -1;
    private int blogHits = -1;
    private String searchString = null;
    private Timestamp fromDate = null;
    private Timestamp toDate = null;
    private String searchIndexDir = null;
    private int hitCount = 0;
    private Collection searchResult = null;
    // 1|2 = 3;
    private int scopeInArticle = SEARCH_ONLY_TITLE | SEARCH_ONLY_BODY;
    private int sort = SEARCH_SORT_DEFAULT;//sort type 1=by key 2=by id
    /** Creates a new instance of ArticleSearchQuery */
    public SearchQuery() {
        searchIndexDir = Sysconfig.getIndexDir();
    }

    /**
     * Set name of the author that should be searched for
     * @param membername
     */
    public void setUserName(String membername) {
        this.userName = membername;
    }

    /**
     * Id of forum where post belongs. Set to -1 if all forums should be searched
     * @param catalogid
     */
    public void setCatalogId(int catalogid) {
        this.catalogid = catalogid;
    }

    /**
     * Set string that should be searched for.
     * @param searchString
     */
    public void setSearchString(String searchString) {
        this.searchString = searchString;
    }

    public void setScopeInArticle(int scopeInArticle) {
        this.scopeInArticle = scopeInArticle;
    }

    public void setArticleHitsLimit(int hitsNum) {
        this.blogHits = hitsNum;
    }

    public void setFromDate(Timestamp fromDate) {
        this.fromDate = fromDate;
    }

    public void setToDate(Timestamp toDate) {
        this.toDate = toDate;
    }

    public void setSortType(int stype) {
        this.sort = stype;
    }
    // Note that with IndexSearcher, the directory is closed automatically
    protected IndexSearcher getSearcher(Directory directory) throws IOException {
        try {
            IndexSearcher searcher = new IndexSearcher(directory);
            return searcher;
        } catch (IOException ex) {
            // we throw new IOException because the original exception
            // contain sensitive directory information
            log.error("Cannot access the lucene search index for query.  You can also go to Admin Zone to rebuild the Lucene index files.", ex);
            //@todo : localize me
            throw new IOException("Cannot access the lucene search index. Please report this error to web site Administrator .");
        }
    }

    public void searchDocuments(int offset, int rowsToReturn) throws IOException {
        // Now check if at least one of these input is present: key, member, attachment
        if (searchString == null || searchString.equals("")) {
            return;
        }
        //Build the query
        BooleanQuery query = new BooleanQuery();
        try {
            Query topicBodyQuery = getTopicBodyQuery();
            if (topicBodyQuery != null) {
                query.add(topicBodyQuery, BooleanClause.Occur.MUST);
                log.debug("topicBodyQuery = " + topicBodyQuery);
            }
            Query catalogQuery = getCatalogQuery();
            if (catalogQuery != null) {
                log.debug("categoryForumQuery = " + catalogQuery);
                query.add(catalogQuery, BooleanClause.Occur.MUST);
            }

            Query memberQuery = getMemberQuery();
            if (memberQuery != null) {
                log.debug("memberQuery = " + memberQuery);
                query.add(memberQuery, BooleanClause.Occur.MUST);
            }
        } catch (ParseException pe) {
            log.error("Cannot parse the search query", pe);
        }
        log.debug("booleanQuery = " + query);

        RangeFilter dateFilter = null;
        //Add date filter if some of dates provided
        if (fromDate != null && toDate != null) {
            dateFilter = new RangeFilter("ModiTime", DateTools.dateToString(fromDate, Resolution.MILLISECOND), DateTools.dateToString(toDate, Resolution.MILLISECOND), true, true);
        } else if (fromDate != null) {
            dateFilter = RangeFilter.More("ModiTime", DateTools.dateToString(fromDate, Resolution.MILLISECOND));
        } else if (toDate != null) {
            dateFilter = RangeFilter.Less("ModiTime", DateTools.dateToString(toDate, Resolution.MILLISECOND));
        }
        //add article visited filter
        IntegerFilter blogHitsFilter = null;
        if (blogHits > 0) {
            blogHitsFilter = IntegerFilter.greaterThan("Hits", blogHits);
        }
        Filter filter = null;
        if (dateFilter != null) {
            if (blogHitsFilter != null) {
                filter = new CombineFilter(dateFilter, blogHitsFilter);
            } else {
                filter = dateFilter;
            }
        } else {
            filter = blogHitsFilter;
        }
        //Now search the documents
        Directory directory = null;
        IndexSearcher searcher = null;
        try {
            directory = SearchService.getSearchIndexDir();
            searcher = getSearcher(directory);
            //If filter set then use it
            Hits postHits = null;
            if (filter != null) {
                postHits = searcher.search(query, filter);
            } else {
                if (filter != null) {
                    postHits = searcher.search(query, filter, getQuerySort());
                } else {
                    postHits = searcher.search(query, getQuerySort());
                }
            }

            hitCount = postHits.length();
            //System.out.print("Offset:"+offset+"hitCount:"+hitCount+"\n");
            searchResult = getArticles(postHits, offset, rowsToReturn);
        } catch (IOException ex) {
            throw ex;
        } finally {
            // NOTE that we don't close directory because searcher.close() already do that
            if (searcher != null) {
                try {
                    searcher.close();
                } catch (IOException ex) {
                    log.debug("Error closing Lucene IndexSearcher", ex);
                }
            }
        }
    }

    public int getHitCount() {
        return hitCount;
    }

    public Collection getArticleResult() {
        if (searchResult == null) {
            //create an empty list, in case result is null
            searchResult = new ArrayList();
        }
        return searchResult;
    }

    private Collection getArticles(Hits postHits, int offset, int rowsToReturn)
            throws IOException {

        if (offset < 0) {
            throw new IllegalArgumentException("The offset < 0 is not allowed.");
        }
        if (rowsToReturn <= 0) {
            throw new IllegalArgumentException("The rowsToReturn <= 0 is not allowed.");
        }

        //int hitCount = getHitCount();
        ArrayList retValue = new ArrayList(hitCount);

        for (int i = offset; (i < offset + rowsToReturn) && (i < hitCount); i++) {
            Document postDocument = postHits.doc(i);
            int artId = Integer.parseInt(postDocument.get("id"));
            DataField df = DaoFactory.getArticleDAO().getArticle(artId);
            retValue.add(df);
        }
        return retValue;
    }

    private Query getTopicBodyQuery() throws ParseException {
        if (searchString == null || searchString.equals("")) {
            return null;
        }
        Analyzer analyzer = ArticleIndex.getAnalyzer();
        BooleanQuery topicBodyQuery = new BooleanQuery();
        //add topic query
        Query topicQuery = new QueryParser("Title", analyzer).parse(searchString);
        topicBodyQuery.add(topicQuery, BooleanClause.Occur.SHOULD);         
        //add body query
        Query bodyQuery = new QueryParser("Content", analyzer).parse(searchString);
        if (scopeInArticle == SEARCH_ONLY_TITLE) {
            return topicQuery;
        } else if (scopeInArticle == SEARCH_ONLY_BODY) {
            return bodyQuery;
        }
        topicBodyQuery.add(bodyQuery, BooleanClause.Occur.SHOULD);
        return topicBodyQuery;
    }

    private Query getMemberQuery() {
        Query memberQuery = null;
        if (userName != null) {
            Term memberTerm = new Term("UserName", userName);
            memberQuery = new TermQuery(memberTerm);
        }
        return memberQuery;
    }

    private Query getCatalogQuery() {
        Query catalogQuery = null;
        if (catalogid > 0) {
            // search in forum
            Term catalogTerm = new Term("CatalogId", String.valueOf(catalogid));
            catalogQuery = new TermQuery(catalogTerm);
        }
        return catalogQuery;
    }

    public void setSort(int sort) {
        if ( (sort != SEARCH_SORT_DEFAULT) && (sort != SEARCH_SORT_TIME_ASC) && (sort != SEARCH_SORT_TIME_DESC)) {
            throw new IllegalArgumentException("Does not support sort = " + sort);
        }
        this.sort = sort;
    }
    public int getSort() {
        return sort;
    }
    private Sort getQuerySort() {
        Sort sortObj = null;
        switch (sort) {
            case SEARCH_SORT_TIME_ASC:
                sortObj = new Sort(new SortField("ModiTime", SortField.STRING, false));
                break;
            case SEARCH_SORT_TIME_DESC:
                sortObj = new Sort(new SortField("ModiTime", SortField.STRING, true));
                break;
            default:
                sortObj = new Sort();
                break;
        }
        return sortObj;
    }
}
