/*
 * Copyright ©2018 vbill.cn.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * </p>
 */

package cn.vbill.middleware.porter.common.util.db;

import org.apache.commons.beanutils.ConversionException;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

/**
 * 
 * @author guohongjian[guo_hj@suixingpay.com]
 *
 */
public class SqlTimestampConverter implements Converter {
    /**
     * Field description
     */
    public static final String[] DATE_FORMATS = new String[] {"yyyy-MM-dd", "HH:mm:ss", "yyyy-MM-dd HH:mm:ss",
        "yyyy-MM-dd:HH:mm:ss", "EEE MMM dd HH:mm:ss zzz yyyy",
        DateFormatUtils.ISO_DATETIME_FORMAT.getPattern(), DateFormatUtils.ISO_DATETIME_TIME_ZONE_FORMAT.getPattern(),
        DateFormatUtils.SMTP_DATETIME_FORMAT.getPattern()};

    /**
     * SQL_TIMESTAMP
     */
    public static final Converter SQL_TIMESTAMP = new SqlTimestampConverter(null);
    private static final Logger LOGGER = LoggerFactory.getLogger(SqlTimestampConverter.class);

    /**
     * The default value specified to our Constructor, if any.
     */
    private final Object defaultValue;

    /**
     * Should we return the default value on conversion errors?
     */
    private final boolean useDefault;

    /**
     * Create a {@link Converter} that will throw a {@link ConversionException} if a conversion error occurs.
     */
    public SqlTimestampConverter() {
        this.defaultValue = null;
        this.useDefault = false;
    }

    /**
     * Create a {@link Converter} that will return the specified default value if a conversion error occurs.
     *
     * @param defaultValue The default value to be returned
     */
    public SqlTimestampConverter(Object defaultValue) {
        this.defaultValue = defaultValue;
        this.useDefault = true;
    }

    /**
     * Convert the specified input object into an output object of the specified type.
     *
     * @param type  Data type to which this value should be converted
     * @param value The input value to be converted
     * @throws ConversionException if conversion cannot be performed successfully
     */
    public Object convert(Class type, Object value) {
        if (value == null) {
            if (useDefault) {
                return (defaultValue);
            } else {
                throw new ConversionException("No value specified");
            }
        }

        if (value instanceof java.sql.Date && java.sql.Date.class.equals(type)) {
            return value;
        } else if (value instanceof java.sql.Time && java.sql.Time.class.equals(type)) {
            return value;
        } else if (value instanceof Timestamp && Timestamp.class.equals(type)) {
            return value;
        } else {
            try {
                if (java.sql.Date.class.equals(type)) {
                    return new java.sql.Date(convertTimestamp2TimeMillis(value.toString()));
                } else if (java.sql.Time.class.equals(type)) {
                    return new java.sql.Time(convertTimestamp2TimeMillis(value.toString()));
                } else if (Timestamp.class.equals(type)) {
                    return new Timestamp(convertTimestamp2TimeMillis(value.toString()));
                } else {
                    return new Timestamp(convertTimestamp2TimeMillis(value.toString()));
                }
            } catch (Exception e) {
                throw new ConversionException("Value format invalid: " + e.getMessage(), e);
            }
        }

    }

    /**
     * convertTimestamp2TimeMillis
     *
     * @param input
     * @return
     */
    private Long convertTimestamp2TimeMillis(String input) {
        if (input == null) {
            return null;
        }

        try {
            // 先处理Timestamp类型
            return Timestamp.valueOf(input).getTime();
        } catch (Exception nfe) {
            try {
                try {
                    return parseDate(input, DATE_FORMATS, Locale.ENGLISH).getTime();
                } catch (Exception err) {
                    return parseDate(input, DATE_FORMATS, Locale.getDefault()).getTime();
                }
            } catch (Exception err) {
                // 最后处理long time的情况
                return Long.parseLong(input);
            }
        }
    }

    /**
     * parseDate
     *
     * @param str
     * @param parsePatterns
     * @param locale
     * @return
     * @throws ParseException
     */
    private Date parseDate(String str, String[] parsePatterns, Locale locale) throws ParseException {
        if ((str == null) || (parsePatterns == null)) {
            throw new IllegalArgumentException("Date and Patterns must not be null");
        }

        SimpleDateFormat parser = null;
        ParsePosition pos = new ParsePosition(0);

        for (int i = 0; i < parsePatterns.length; i++) {
            if (i == 0) {
                parser = new SimpleDateFormat(parsePatterns[0], locale);
            } else {
                parser.applyPattern(parsePatterns[i]);
            }
            pos.setIndex(0);
            Date date = parser.parse(str, pos);
            if ((date != null) && (pos.getIndex() == str.length())) {
                return date;
            }
        }

        throw new ParseException("Unable to parse the date: " + str, -1);
    }
}