<?php

// +----------------------------------------------------------------------
// | ThinkPHP [ WE CAN DO IT JUST THINK IT ]
// +----------------------------------------------------------------------
// | Copyright (c) 2009 http://thinkphp.cn All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: liu21st <liu21st@gmail.com>
// +----------------------------------------------------------------------

/**
 * 图像操作类库
 * @category   ORG
 * @package  ORG
 * @subpackage  Util
 * @author    liu21st <liu21st@gmail.com>
 */
define('IMAGE_FONT', LIB_PATH . 'Font' . DS . 'simsun.ttc');   //定义字体路径(完整路径及文件名)

class Image {

    /**
     * 取得图像信息
     * @static
     * @access public
     * @param string $image 图像文件名
     * @return mixed
     */
    static function getImageInfo($img) {
        $imageInfo = getimagesize($img);
        if ($imageInfo !== false) {
            $imageType = strtolower(substr(image_type_to_extension($imageInfo[2]), 1));
            $imageSize = filesize($img);
            $info = array(
                "width" => $imageInfo[0],
                "height" => $imageInfo[1],
                "type" => $imageType,
                "size" => $imageSize,
                "mime" => $imageInfo['mime']
            );
            return $info;
        } else {
            return false;
        }
    }

    /**
     * 为图片添加水印
     * @static public
     * @param string $source 原文件名
     * @param string $water  水印图片
     * @param string $$savename  添加水印后的图片名
     * @param string $alpha  水印的透明度
     * @return void
     */
    static public function water($source, $water, $savename = null, $alpha = 80) {
        //检查文件是否存在
        if (!file_exists($source) || !file_exists($water))
            return false;

        //图片信息
        $sInfo = self::getImageInfo($source);
        $wInfo = self::getImageInfo($water);

        //如果图片小于水印图片，不生成图片
        if ($sInfo["width"] < $wInfo["width"] || $sInfo['height'] < $wInfo['height'])
            return false;

        //建立图像
        $sCreateFun = "imagecreatefrom" . $sInfo['type'];
        $sImage = $sCreateFun($source);
        $wCreateFun = "imagecreatefrom" . $wInfo['type'];
        $wImage = $wCreateFun($water);

        //设定图像的混色模式
        imagealphablending($wImage, true);

        //图像位置,默认为右下角右对齐
        $posY = $sInfo["height"] - $wInfo["height"];
        $posX = $sInfo["width"] - $wInfo["width"];

        //生成混合图像
        imagecopymerge($sImage, $wImage, $posX, $posY, 0, 0, $wInfo['width'], $wInfo['height'], $alpha);

        //输出图像
        $ImageFun = 'Image' . $sInfo['type'];
        //如果没有给出保存文件名，默认为原图像名
        if (!$savename) {
            $savename = $source;
            @unlink($source);
        }
        //保存图像
        $ImageFun($sImage, $savename);
        imagedestroy($sImage);
    }

    /**
     * 为图片添加水印
     * @param type $source          需要处理的图片
     * @param type $water           水印图片
     * @param type $conditionWidth  当满足$sourde图片宽度大于此值时才添加水印
     * @param type $conditionHeight 当满足$sourde图片高度大于此值时才添加水印
     * @param type $alpha           水印透明度
     * @param type $quality         生成图片质量
     * @param type $position        水印位置(1:左上；2：中上；3：右上；4：左中；5：正中；6：右中；7：左下；8：中下；9：右下)
     * @param type $imageSavePath   保存路径（为空则覆盖原图片,不包含文件名）
     */
    static public function bpiWater($source, $water, $conditionWidth, $conditionHeight, $alpha, $quality, $position, $imageSavePath = null) {
        //检查文件是否存在
        if (!file_exists($source) || !file_exists($water)) {
            return false;
        }

        //参数处理
        $conditionWidth = intval($conditionWidth);
        $conditionHeight = intval($conditionHeight);
        $alpha = intval($alpha);
        $quality = intval($quality);
        $position = intval($position);

        //图片信息
        $sourceInfo = self::getImageInfo($source);
        $waterInfo = self::getImageInfo($water);

        if (!$sourceInfo || !$waterInfo) {
            return false;
        }

        $type = $sourceInfo['type'] == 'jpg' ? 'jpeg' : $sourceInfo['type'];

        //如果没有给出保存文件名，默认为原图像名
        if (!$imageSavePath) {
            $savePath = dirname($source) . DS;
            $savaName = basename($source);
        } else {
            $savePath = rtrim($imageSavePath, '\\/') . DS;
            if (!is_dir($savePath)) {
                mkdir($savePath, 0777, true);
            }
            $savaName = uniqid() . '.' . $type;
        }

        //gif 图像不生成水印
        //如果图片小于水印图片，或者未达到条件则不生成图片
        if (($type == 'gif') || ($sourceInfo["width"] < $waterInfo["width"] || $sourceInfo['height'] < $waterInfo['height']) || ($sourceInfo["width"] < $conditionWidth || $sourceInfo['height'] < $conditionHeight)) {
            $data = array(
                'savePath' => $savePath,
                'saveName' => $savaName,
                'ext' => $type,
                'size' => filesize($savePath . $savaName)
            );
            if (!$imageSavePath) {
                return $data;
            } else {
                $status = copy($source, $savePath . $savaName);
                return $status ? $data : false;
            }
        }

        //水印位置
        switch ($position) {
            case 1:
                //左上角
                $posX = 0;
                $posY = 0;
                break;
            case 2:
                //中上
                $posX = intval(($sourceInfo["width"] - $waterInfo["width"]) / 2);
                $posY = 0;
                break;
            case 3:
                //右上角
                $posX = $sourceInfo["width"] - $waterInfo["width"];
                $posY = 0;
                break;
            case 4:
                //左中
                $posX = 0;
                $posY = intval(($sourceInfo["height"] - $waterInfo["height"]) / 2);
                break;
            case 5:
                //正中
                $posX = intval(($sourceInfo["width"] - $waterInfo["width"]) / 2);
                $posY = intval(($sourceInfo["height"] - $waterInfo["height"]) / 2);
                break;
            case 6:
                //右中
                $posX = $sourceInfo["width"] - $waterInfo["width"];
                $posY = intval(($sourceInfo["height"] - $waterInfo["height"]) / 2);
                break;
            case 7:
                //左下
                $posX = 0;
                $posY = $sourceInfo["height"] - $waterInfo["height"];
                break;
            case 8:
                //中下
                $posX = intval(($sourceInfo["width"] - $waterInfo["width"]) / 2);
                $posY = $sourceInfo["height"] - $waterInfo["height"];
                break;
            default:
                //默认右下角
                $posX = $sourceInfo["width"] - $waterInfo["width"];
                $posY = $sourceInfo["height"] - $waterInfo["height"];
                break;
        }

        //建立画布
        $sourceCreateFun = "imagecreatefrom" . $type;
        $sourceImage = $sourceCreateFun($source);
        $waterCreateFun = "imagecreatefrom" . ($waterInfo['type'] == 'jpg' ? 'jpeg' : $waterInfo['type'] );
        $waterImage = $waterCreateFun($water);

        //设定图像的混色模式
        imagealphablending($waterImage, true);

        //生成混合图像
        //水印为PNG时需要特别处理
        if ($waterInfo['type'] === 'png') {
            $waterImageTemp = imagecreatetruecolor($waterInfo['width'], $waterInfo['height']);
            imagecopy($waterImageTemp, $sourceImage, 0, 0, $posX, $posY, $waterInfo['width'], $waterInfo['height']);
            imagecopy($waterImageTemp, $waterImage, 0, 0, 0, 0, $waterInfo['width'], $waterInfo['height']);
            $waterImage = $waterImageTemp;
        }
        imagecopymerge($sourceImage, $waterImage, $posX, $posY, 0, 0, $waterInfo['width'], $waterInfo['height'], $alpha);

        if (!$imageSavePath) {
            @unlink($source);
        }
        //保存图像
        if ($type == 'jpeg') {
            imageinterlace($sourceImage, true);
            imagejpeg($sourceImage, $savePath . $savaName, $quality);
        } else {
            //图像处理函数
            $ImageFun = 'image' . $type;
            if (!function_exists($ImageFun)) {
                return false;
            }
            $ImageFun($sourceImage, $savePath . $savaName);
        }
        imagedestroy($sourceImage);
        imagedestroy($waterImage);
        isset($waterImageTemp) && imagedestroy($waterImageTemp);
        $data = array(
            'savePath' => $savePath,
            'saveName' => $savaName,
            'ext' => $type,
            'size' => filesize($savePath . $savaName)
        );
        return $data;
    }

    /**
     * 生成缩略图
     * @static
     * @access public
     * @param string $image  原图
     * @param string $type 图像格式
     * @param string $thumbname 缩略图文件名
     * @param string $maxWidth  宽度
     * @param string $maxHeight  高度
     * @param string $position 缩略图保存目录
     * @param boolean $interlace 启用隔行扫描
     * @return void
     */
    static function thumb($image, $thumbname, $type = '', $maxWidth = 200, $maxHeight = 50, $interlace = true) {
        // 获取原图信息
        $info = Image::getImageInfo($image);
        if ($info !== false) {
            $srcWidth = $info['width'];
            $srcHeight = $info['height'];
            $type = empty($type) ? $info['type'] : $type;
            $type = strtolower($type);
            $interlace = $interlace ? 1 : 0;
            unset($info);
            $scale = min($maxWidth / $srcWidth, $maxHeight / $srcHeight); // 计算缩放比例
            if ($scale >= 1) {
                // 超过原图大小不再缩略
                $width = $srcWidth;
                $height = $srcHeight;
            } else {
                // 缩略图尺寸
                $width = (int) ($srcWidth * $scale);
                $height = (int) ($srcHeight * $scale);
            }

            // 载入原图
            $createFun = 'ImageCreateFrom' . ($type == 'jpg' ? 'jpeg' : $type);
            if (!function_exists($createFun)) {
                return false;
            }
            $srcImg = $createFun($image);

            //创建缩略图
            if ($type != 'gif' && function_exists('imagecreatetruecolor'))
                $thumbImg = imagecreatetruecolor($width, $height);
            else
                $thumbImg = imagecreate($width, $height);
            //png和gif的透明处理 by luofei614
            if ('png' == $type) {
                imagealphablending($thumbImg, false); //取消默认的混色模式（为解决阴影为绿色的问题）
                imagesavealpha($thumbImg, true); //设定保存完整的 alpha 通道信息（为解决阴影为绿色的问题）    
            } elseif ('gif' == $type) {
                $trnprt_indx = imagecolortransparent($srcImg);
                if ($trnprt_indx >= 0) {
                    //its transparent
                    $trnprt_color = imagecolorsforindex($srcImg, $trnprt_indx);
                    $trnprt_indx = imagecolorallocate($thumbImg, $trnprt_color['red'], $trnprt_color['green'], $trnprt_color['blue']);
                    imagefill($thumbImg, 0, 0, $trnprt_indx);
                    imagecolortransparent($thumbImg, $trnprt_indx);
                }
            }
            // 复制图片
            if (function_exists("ImageCopyResampled"))
                imagecopyresampled($thumbImg, $srcImg, 0, 0, 0, 0, $width, $height, $srcWidth, $srcHeight);
            else
                imagecopyresized($thumbImg, $srcImg, 0, 0, 0, 0, $width, $height, $srcWidth, $srcHeight);

            // 对jpeg图形设置隔行扫描
            if ('jpg' == $type || 'jpeg' == $type)
                imageinterlace($thumbImg, $interlace);

            // 生成图片
            $imageFun = 'image' . ($type == 'jpg' ? 'jpeg' : $type);
            $imageFun($thumbImg, $thumbname);
            imagedestroy($thumbImg);
            imagedestroy($srcImg);
            return $thumbname;
        }
        return false;
    }

    /**
     * 生成 size*size 大小的缩略图
     * @param type $source      原图片地址
     * @param type $thumbPath   缩略图存放路径（绝对路径包含文件名）
     * @param type $size        缩略图大小
     * @param type $type        缩略图格式
     * @param type $interlace   是否开启隔行扫描
     * @return boolean
     */
    static function equalRatioThumb($source, $thumbPath, $size = 180, $type = 'jpg', $interlace = true) {
        if (empty($source) || empty($thumbPath)) {
            return false;
        }
        $sourceInfo = Image::getImageInfo($source);
        if ($sourceInfo === false) {
            return false;
        }
        $sourceWidth = $sourceInfo['width'];
        $sourceHeight = $sourceInfo['height'];

        // 载入原图
        $createFun = 'ImageCreateFrom' . $sourceInfo['type'];
        if (!function_exists($createFun)) {
            return false;
        }
        $sourceImage = $createFun($source);

        //创建缩略图
        if ($type != 'gif' && function_exists('imagecreatetruecolor')) {
            $thumbImage = imagecreatetruecolor($size, $size);
        } else {
            $thumbImage = imagecreate($size, $size);
        }

        //png和gif的透明处理
        if ('png' == $type) {
            imagealphablending($thumbImage, false); //取消默认的混色模式（为解决阴影为绿色的问题）
            imagesavealpha($thumbImage, true); //设定保存完整的 alpha 通道信息（为解决阴影为绿色的问题）    
        } else if ('gif' == $type) {
            $trnprtIndx = imagecolortransparent($sourceImage);
            if ($trnprtIndx >= 0) {
                $trnprtColor = imagecolorsforindex($sourceImage, $trnprtIndx);
                $trnprtIndx = imagecolorallocate($thumbImage, $trnprtColor['red'], $trnprtColor['green'], $trnprtColor['blue']);
                imagefill($thumbImage, 0, 0, $trnprtIndx);
                imagecolortransparent($thumbImage, $trnprtIndx);
            }
        }

        if (function_exists("imageCopyResampled")) {
            imagecopyresampled($thumbImage, $sourceImage, 0, 0, 0, 0, $size, $size, $sourceWidth, $sourceHeight);
        } else {
            imagecopyresized($thumbImage, $sourceImage, 0, 0, 0, 0, $size, $size, $sourceWidth, $sourceHeight);
        }

        // 生成图片
        $imageFun = 'image' . ($type == 'jpg' ? 'jpeg' : $type);
        $path = dirname($thumbPath);
        if (!is_dir($path)) {
            mkdir($path, 0777, true);
        }
        // 对jpeg图形设置隔行扫描
        if ('jpg' == $type || 'jpeg' == $type) {
            imageinterlace($thumbImage, $interlace);
            imagejpeg($thumbImage, $thumbPath, 100);
        } else {
            $imageFun($thumbImage, $thumbPath);
        }
        imagedestroy($thumbImage);
        imagedestroy($sourceImage);
        return true;
    }

    /**
     * 裁剪图片
     * @param type $image       源图片路径
     * @param type $savePath    裁剪图保存目录
     * @param type $startX      裁剪开始横坐标
     * @param type $startY      裁剪开始纵坐标
     * @param type $cropWidth   裁剪宽度
     * @param type $cropHeight  裁剪长度
     * @param type $type        裁剪图保存格式
     * @param type $interlace   是否开启隔行扫描
     * @return boolean
     */
    static function crop($image, $savePath, $startX, $startY, $cropWidth, $cropHeight, $type = '', $interlace = true) {
        // 获取原图信息
        $info = Image::getImageInfo($image);
        if (!$info) {
            return false;
        }
        $srcWidth = $info['width'];
        $srcHeight = $info['height'];
        $srcType = $info['type'];
        $type = empty($type) ? $srcType : $type;
        $interlace = $interlace ? 1 : 0;
        unset($info);

        $startX = intval($startX);
        $startY = intval($startY);
        $cropWidth = intval($cropWidth);
        $cropHeight = intval($cropHeight);
        if ($startX >= $srcWidth || $startY >= $srcHeight || $cropWidth == 0 || $cropHeight == 0) {
            return false;
        }
        if ($startX + $cropWidth > $srcWidth) {
            $cropWidth = $srcWidth - $startX;
        }
        if ($startY + $cropHeight > $srcHeight) {
            $cropHeight = $srcHeight - $startY;
        }

        //载入原图
        $createFun = 'imagecreatefrom' . ($srcType == 'jpg' ? 'jpeg' : $srcType);
        $imageSource = $createFun($image);
        //生成裁剪图画布
        if ($type != 'gif' && function_exists('imagecreatetruecolor')) {
            $cropSource = imagecreatetruecolor($cropWidth, $cropHeight);
        } else {
            $cropSource = imagecreate($cropWidth, $cropHeight);
        }
        //复制区域图像到裁剪图画布
        if (function_exists("imagecopyresampled")) {
            imagecopyresampled($cropSource, $imageSource, 0, 0, $startX, $startY, $cropWidth, $cropHeight, $cropWidth, $cropHeight);
        } else {
            imagecopyresized($cropSource, $imageSource, 0, 0, $startX, $startY, $cropWidth, $cropHeight, $cropWidth, $cropHeight);
        }

        //保存裁剪图
        $imageFun = 'image' . ($type == 'jpg' ? 'jpeg' : $type);
        if (!function_exists($imageFun)) {
            return false;
        }
        $savePath = rtrim($savePath, '\\/') . DS;
        if (!is_dir($savePath)) {
            mkdir($savePath, 0777, true);
        }
        $savaName = uniqid() . '.' . $type;
        // 对jpeg图形设置隔行扫描
        if ('jpg' == $type || 'jpeg' == $type) {
            imageinterlace($cropSource, $interlace);
            imagejpeg($cropSource, $savePath . $savaName, 100);
        } else {
            $imageFun($cropSource, $savePath . $savaName);
        }
        imagedestroy($imageSource);
        imagedestroy($cropSource);

        $data = array();
        $data['savePath'] = $savePath;
        $data['saveName'] = $savaName;
        $data['ext'] = $type;
        $data['size'] = filesize($savePath . $savaName);
        return $data;
    }

    /**
     * 根据给定的字符串生成图像
     * @static
     * @access public
     * @param string $string  字符串
     * @param string $size  图像大小 width,height 或者 array(width,height)
     * @param string $font  字体信息 fontface,fontsize 或者 array(fontface,fontsize)
     * @param string $type 图像格式 默认PNG
     * @param integer $disturb 是否干扰 1 点干扰 2 线干扰 3 复合干扰 0 无干扰
     * @param bool $border  是否加边框 array(color)
     * @return string
     */
    static function buildString($string, $rgb = array(), $filename = '', $type = 'png', $disturb = 1, $border = true) {
        if (is_string($size))
            $size = explode(',', $size);
        $width = $size[0];
        $height = $size[1];
        if (is_string($font))
            $font = explode(',', $font);
        $fontface = $font[0];
        $fontsize = $font[1];
        $length = strlen($string);
        $width = ($length * 9 + 10) > $width ? $length * 9 + 10 : $width;
        $height = 22;
        if ($type != 'gif' && function_exists('imagecreatetruecolor')) {
            $im = @imagecreatetruecolor($width, $height);
        } else {
            $im = @imagecreate($width, $height);
        }
        if (empty($rgb)) {
            $color = imagecolorallocate($im, 102, 104, 104);
        } else {
            $color = imagecolorallocate($im, $rgb[0], $rgb[1], $rgb[2]);
        }
        $backColor = imagecolorallocate($im, 255, 255, 255);    //背景色（随机）
        $borderColor = imagecolorallocate($im, 100, 100, 100);                    //边框色
        $pointColor = imagecolorallocate($im, mt_rand(0, 255), mt_rand(0, 255), mt_rand(0, 255));                 //点颜色

        @imagefilledrectangle($im, 0, 0, $width - 1, $height - 1, $backColor);
        @imagerectangle($im, 0, 0, $width - 1, $height - 1, $borderColor);
        @imagestring($im, 5, 5, 3, $string, $color);
        if (!empty($disturb)) {
            // 添加干扰
            if ($disturb = 1 || $disturb = 3) {
                for ($i = 0; $i < 25; $i++) {
                    imagesetpixel($im, mt_rand(0, $width), mt_rand(0, $height), $pointColor);
                }
            } elseif ($disturb = 2 || $disturb = 3) {
                for ($i = 0; $i < 10; $i++) {
                    imagearc($im, mt_rand(-10, $width), mt_rand(-10, $height), mt_rand(30, 300), mt_rand(20, 200), 55, 44, $pointColor);
                }
            }
        }
        Image::output($im, $type, $filename);
    }

    /**
     * 生成图像验证码
     * @static
     * @access public
     * @param string $length  位数
     * @param string $mode  类型
     * @param string $type 图像格式
     * @param string $width  宽度
     * @param string $height  高度
     * @return string
     */
    static function buildImageVerify($length = 4, $mode = 1, $type = 'png', $width = 48, $height = 22, $verifyName = 'verify') {
        //修改路径
        //import('ORG.Util.String');
        import('String');
        $randval = String::randString($length, $mode);
        //优化处理手段
        $verifyList = session('_verify_');
        $verifyList[$verifyName] = md5($randval);
        session('_verify_', $verifyList);
        unset($verifyList);
        $width = ($length * 10 + 10) > $width ? $length * 10 + 10 : $width;
        if ($type != 'gif' && function_exists('imagecreatetruecolor')) {
            $im = imagecreatetruecolor($width, $height);
        } else {
            $im = imagecreate($width, $height);
        }
        $r = Array(225, 255, 255, 223);
        $g = Array(225, 236, 237, 255);
        $b = Array(225, 236, 166, 125);
        $key = mt_rand(0, 3);

        $backColor = imagecolorallocate($im, $r[$key], $g[$key], $b[$key]);    //背景色（随机）
        $borderColor = imagecolorallocate($im, 100, 100, 100);                    //边框色
        imagefilledrectangle($im, 0, 0, $width - 1, $height - 1, $backColor);
        imagerectangle($im, 0, 0, $width - 1, $height - 1, $borderColor);
        $stringColor = imagecolorallocate($im, mt_rand(0, 200), mt_rand(0, 120), mt_rand(0, 120));
        // 干扰
        for ($i = 0; $i < 10; $i++) {
            imagearc($im, mt_rand(-10, $width), mt_rand(-10, $height), mt_rand(30, 300), mt_rand(20, 200), 55, 44, $stringColor);
        }
        for ($i = 0; $i < 25; $i++) {
            imagesetpixel($im, mt_rand(0, $width), mt_rand(0, $height), $stringColor);
        }
        $start = ($width - $length * 10) / 2;
        for ($i = 0; $i < $length; $i++) {
            imagestring($im, 5, $start + $i * 10 + 3, mt_rand(8, 16), $randval{$i}, $stringColor);
        }
        Image::output($im, $type);
    }

    // 中文验证码
    static function GBVerify($length = 4, $type = 'png', $width = 180, $height = 50, $fontface = 'simhei.ttf', $verifyName = 'verify') {
        //修改路径
        //import('ORG.Util.String');
        import('String');
        $code = String::randString($length, 4);
        $width = ($length * 45) > $width ? $length * 45 : $width;
        session($verifyName, md5($code));
        $im = imagecreatetruecolor($width, $height);
        $borderColor = imagecolorallocate($im, 100, 100, 100);                    //边框色
        $bkcolor = imagecolorallocate($im, 250, 250, 250);
        imagefill($im, 0, 0, $bkcolor);
        @imagerectangle($im, 0, 0, $width - 1, $height - 1, $borderColor);
        // 干扰
        for ($i = 0; $i < 15; $i++) {
            $fontcolor = imagecolorallocate($im, mt_rand(0, 255), mt_rand(0, 255), mt_rand(0, 255));
            imagearc($im, mt_rand(-10, $width), mt_rand(-10, $height), mt_rand(30, 300), mt_rand(20, 200), 55, 44, $fontcolor);
        }
        for ($i = 0; $i < 255; $i++) {
            $fontcolor = imagecolorallocate($im, mt_rand(0, 255), mt_rand(0, 255), mt_rand(0, 255));
            imagesetpixel($im, mt_rand(0, $width), mt_rand(0, $height), $fontcolor);
        }
        if (!is_file($fontface)) {
            $fontface = dirname(__FILE__) . "/" . $fontface;
        }
        for ($i = 0; $i < $length; $i++) {
            $fontcolor = imagecolorallocate($im, mt_rand(0, 120), mt_rand(0, 120), mt_rand(0, 120)); //这样保证随机出来的颜色较深。
            $codex = String::msubstr($code, $i, 1);
            imagettftext($im, mt_rand(16, 20), mt_rand(-60, 60), 40 * $i + 20, mt_rand(30, 35), $fontcolor, $fontface, $codex);
        }
        Image::output($im, $type);
    }

    /**
     * 把图像转换成字符显示
     * @static
     * @access public
     * @param string $image  要显示的图像
     * @param string $type  图像类型，默认自动获取
     * @return string
     */
    static function showASCIIImg($image, $string = '', $type = '') {
        $info = Image::getImageInfo($image);
        if ($info !== false) {
            $type = empty($type) ? $info['type'] : $type;
            unset($info);
            // 载入原图
            $createFun = 'ImageCreateFrom' . ($type == 'jpg' ? 'jpeg' : $type);
            $im = $createFun($image);
            $dx = imagesx($im);
            $dy = imagesy($im);
            $i = 0;
            $out = '<span style="padding:0px;margin:0;line-height:100%;font-size:1px;">';
            set_time_limit(0);
            for ($y = 0; $y < $dy; $y++) {
                for ($x = 0; $x < $dx; $x++) {
                    $col = imagecolorat($im, $x, $y);
                    $rgb = imagecolorsforindex($im, $col);
                    $str = empty($string) ? '*' : $string[$i++];
                    $out .= sprintf('<span style="margin:0px;color:#%02x%02x%02x">' . $str . '</span>', $rgb['red'], $rgb['green'], $rgb['blue']);
                }
                $out .= "<br>\n";
            }
            $out .= '</span>';
            imagedestroy($im);
            return $out;
        }
        return false;
    }

    /**
     * 生成UPC-A条形码
     * @static
     * @param string $type 图像格式
     * @param string $type 图像格式
     * @param string $lw  单元宽度
     * @param string $hi   条码高度
     * @return string
     */
    static function UPCA($code, $type = 'png', $lw = 2, $hi = 100) {
        static $Lencode = array('0001101', '0011001', '0010011', '0111101', '0100011',
    '0110001', '0101111', '0111011', '0110111', '0001011');
        static $Rencode = array('1110010', '1100110', '1101100', '1000010', '1011100',
    '1001110', '1010000', '1000100', '1001000', '1110100');
        $ends = '101';
        $center = '01010';
        /* UPC-A Must be 11 digits, we compute the checksum. */
        if (strlen($code) != 11) {
            die("UPC-A Must be 11 digits.");
        }
        /* Compute the EAN-13 Checksum digit */
        $ncode = '0' . $code;
        $even = 0;
        $odd = 0;
        for ($x = 0; $x < 12; $x++) {
            if ($x % 2) {
                $odd += $ncode[$x];
            } else {
                $even += $ncode[$x];
            }
        }
        $code.= ( 10 - (($odd * 3 + $even) % 10)) % 10;
        /* Create the bar encoding using a binary string */
        $bars = $ends;
        $bars.=$Lencode[$code[0]];
        for ($x = 1; $x < 6; $x++) {
            $bars.=$Lencode[$code[$x]];
        }
        $bars.=$center;
        for ($x = 6; $x < 12; $x++) {
            $bars.=$Rencode[$code[$x]];
        }
        $bars.=$ends;
        /* Generate the Barcode Image */
        if ($type != 'gif' && function_exists('imagecreatetruecolor')) {
            $im = imagecreatetruecolor($lw * 95 + 30, $hi + 30);
        } else {
            $im = imagecreate($lw * 95 + 30, $hi + 30);
        }
        $fg = ImageColorAllocate($im, 0, 0, 0);
        $bg = ImageColorAllocate($im, 255, 255, 255);
        ImageFilledRectangle($im, 0, 0, $lw * 95 + 30, $hi + 30, $bg);
        $shift = 10;
        for ($x = 0; $x < strlen($bars); $x++) {
            if (($x < 10) || ($x >= 45 && $x < 50) || ($x >= 85)) {
                $sh = 10;
            } else {
                $sh = 0;
            }
            if ($bars[$x] == '1') {
                $color = $fg;
            } else {
                $color = $bg;
            }
            ImageFilledRectangle($im, ($x * $lw) + 15, 5, ($x + 1) * $lw + 14, $hi + 5 + $sh, $color);
        }
        /* Add the Human Readable Label */
        ImageString($im, 4, 5, $hi - 5, $code[0], $fg);
        for ($x = 0; $x < 5; $x++) {
            ImageString($im, 5, $lw * (13 + $x * 6) + 15, $hi + 5, $code[$x + 1], $fg);
            ImageString($im, 5, $lw * (53 + $x * 6) + 15, $hi + 5, $code[$x + 6], $fg);
        }
        ImageString($im, 4, $lw * 95 + 17, $hi - 5, $code[11], $fg);
        /* Output the Header and Content. */
        Image::output($im, $type);
    }

    static function output($im, $type = 'png', $filename = '') {
        header("Content-type: image/" . $type);
        $ImageFun = 'image' . $type;
        if (empty($filename)) {
            $ImageFun($im);
        } else {
            $ImageFun($im, $filename);
        }
        imagedestroy($im);
    }

    /**
     * 绘制字符串到图像
     * @param type $source      原图像路径
     * @param type $string      绘制字符串
     * @param type $x           绘制X坐标
     * @param type $y           绘制Y坐标
     * @param type $margin      字符间隔
     * @param type $imageSavePath    保存路径(为空将覆盖原文件, 不含文件名)
     * @param type $straight    是否直接绘制字符串到图像（设置为true则$margin值无效）
     * @param type $fontSize    绘制字符串大小（单位磅(GD2) 单位PX(GD1)）
     * @param type $quality     jpeg文件质量
     * @return boolean
     */
    static function drawString($source, $string, $x = 0, $y = 0, $margin = 12, $imageSavePath = null, $straight = true, $fontSize = 14, $quality = '100') {
        $font = IMAGE_FONT;   //绘制字体
        //检查文件是否存在
        if (!file_exists($source) || !file_exists($font)) {
            return false;
        }
        //图片信息
        $sourceInfo = self::getImageInfo($source);
        if (!$sourceInfo) {
            return false;
        }
        //参数整理
        $x = intval($x);
        $posY = intval($y + $fontSize);
        $margin = intval($margin);
        $quality = intval($quality);
        $type = $sourceInfo['type'] == 'jpg' ? 'jpeg' : $sourceInfo['type'];

        //建立画布
        $sourceCreateFun = "imagecreatefrom" . $type;
        if (!function_exists($sourceCreateFun)) {
            return false;
        }
        $sourceImage = $sourceCreateFun($source);
        $length = utf8StringLength($string);        //字符串长度
        $color = imagecolorallocate($sourceImage, 0, 0, 0);     //绘制字符串颜色
        if ($straight) {
            imagettftext($sourceImage, $fontSize, 0, $x, $posY, $color, $font, $string);
        } else {
            for ($i = 0; $i < $length; $i++) {
                $code = cutString($string, $i, 1);
                $posX = ($i * $margin) + $x;
                imagettftext($sourceImage, $fontSize, 0, $posX, $posY, $color, $font, $code);
            }
        }
        //如果没有给出保存文件名，默认为原图像名
        if (!$imageSavePath) {
            $savePath = dirname($source) . DS;
            $savaName = basename($source);
            @unlink($source);
        } else {
            $savePath = rtrim($imageSavePath, '\\/') . DS;
            if (!is_dir($savePath)) {
                mkdir($savePath, 0777, true);
            }
            $savaName = uniqid() . '.' . $type;
        }

        //保存图像
        if ($type == 'jpeg') {
            imageinterlace($sourceImage, true);
            imagejpeg($sourceImage, $savePath . $savaName, $quality);
        } else {
            //图像处理函数
            $imageFun = 'image' . $type;
            if (!function_exists($imageFun)) {
                return false;
            }
            $imageFun($sourceImage, $savePath . $savaName);
        }
        imagedestroy($sourceImage);

        $data = array(
            'savePath' => $savePath,
            'saveName' => $savaName,
            'ext' => $type,
            'size' => filesize($savePath . $savaName)
        );
        return $data;
    }

    /**
     * 根据字符串长短和填充最大宽度调整字符大小绘制在图片上(字符最小为10pt,最大为14pt)
     * @param type $source          图像源文件
     * @param type $string          字符串
     * @param type $x               X坐标
     * @param type $y               Y坐标
     * @param type $maxWidth        填充最大宽度
     * @param type $minFontSize     字体最小像素
     * @param type $maxFontSize     字体最大像素
     * @param type $imageSavePath   保存名字（为空覆盖原文件，不含文件名）
     * @param type $quality         JPEG图像质量
     */
    static function simpleDrawString($source, $string, $x, $y, $maxWidth, $minFontSize = 10, $maxFontSize = 14, $imageSavePath = null, $quality = '100') {
        $maxWidth = intval($maxWidth);
        $font = IMAGE_FONT;
        if (!file_exists($font)) {
            return false;
        }
        $minFontSize = $fontSize = intval($minFontSize);
        $maxFontSize = intval($maxFontSize);
        for ($i = $maxFontSize; $i >= $minFontSize; $i--) {
            $fontSize = $i;
            $box = imagettfbbox($fontSize, 0, $font, $string);
            $width = abs($box[4] - $box[0]);
            if ($width <= $maxWidth) {
                break;
            }
        }
        return Image::drawString($source, $string, $x, $y, 0, $imageSavePath, true, $fontSize, $quality);
    }

    /**
     * 绘制字符串到图像（支持自动、手动换行）
     * @param type $source      图片原文件路径
     * @param type $string      绘制字符串
     * @param type $x           绘制X坐标
     * @param type $y           绘制Y坐标
     * @param type $maxWidth    绘制最大宽度（为0表示不限制）
     * @param type $maxHeight   绘制最大高度(为0表示不限制)
     * @param type $lineHeight  字符大小像素
     * @param type $lineHeight  字符串行高
     * @param type $imageSavePath    保存路径(不含文件名，为空则覆盖原文件)
     * @param type $quality     jpeg图片质量
     * @return boolean
     */
    static function advancedDrawString($source, $string, $x, $y, $maxWidth, $maxHeight, $fontSize = 12, $lineHeight = 24, $imageSavePath = null, $quality = '100') {
        $font = IMAGE_FONT;
        if (!file_exists($source) || !file_exists($font)) {
            return false;
        }

        //图片信息
        $sourceInfo = self::getImageInfo($source);
        if (!$sourceInfo) {
            return false;
        }

        //参数整理
        $x = intval($x);
        $y = intval($y);
        $maxWidth = intval($maxWidth);
        $maxHeight = intval($maxHeight);
        $fontSize = intval($fontSize);
        $lineHeight = intval($lineHeight);
        $quality = intval($quality);
        $type = $sourceInfo['type'] == 'jpg' ? 'jpeg' : $sourceInfo['type'];

        //字符串换行处理
        $split = "\r\n";        //换行分隔符
        //分割字符串
        $stringArray = explode($split, $string);
        $lineString = array();
        if ($maxWidth != 0) {
            $currentLine = 0;
            foreach ($stringArray as $k => $str) {
                $length = utf8StringLength($str);
                for ($i = 0; $i < $length; $i++) {
                    $code = cutString($str, $i, 1);
                    $box = imagettfbbox($fontSize, 0, $font, $lineString[$currentLine] . $code);
                    if ($box[2] - $box[0] <= $maxWidth) {
                        $lineString[$currentLine] .= $code;
                    } else {
                        $currentLine++;
                        $lineString[$currentLine] = $code;
                    }
                }
                $currentLine++;
            }
        } else {
            $lineString = $stringArray;
        }

        //建立画布
        $sourceCreateFun = "imagecreatefrom" . $type;
        if (!function_exists($sourceCreateFun)) {
            return false;
        }
        $sourceImage = $sourceCreateFun($source);
        $color = imagecolorallocate($sourceImage, 0, 0, 0); //绘制字符串颜色
        foreach ($lineString as $k => $str) {
            if ($maxHeight != 0 && ($k + 1) * $lineHeight > $maxHeight) {
                break;
            }
            $y += $lineHeight;
            imagettftext($sourceImage, $fontSize, 0, $x, $y, $color, $font, $str);
        }
        //如果没有给出保存文件名，默认为原图像名
        if (!$imageSavePath) {
            $savePath = dirname($source) . DS;
            $savaName = basename($source);
            @unlink($source);
        } else {
            $savePath = rtrim($imageSavePath, '\\/') . DS;
            if (!is_dir($savePath)) {
                mkdir($savePath, 0777, true);
            }
            $savaName = uniqid() . '.' . $type;
        }
        //保存图像
        if ($type == 'jpeg') {
            imageinterlace($sourceImage, true);
            imagejpeg($sourceImage, $savePath . $savaName, 100);
        } else {
            //图像处理函数
            $imageFun = 'image' . $type;
            if (!function_exists($imageFun)) {
                return false;
            }
            $imageFun($sourceImage, $savePath . $savaName);
        }
        imagedestroy($sourceImage);
        $data = array(
            'savePath' => $savePath,
            'saveName' => $savaName,
            'ext' => $type,
            'size' => filesize($savePath . $savaName)
        );
        return $data;
    }

}