<?php

/**
 * validate.service
 *
 * 数据验证服务类
 *
 * @package service
 * @author awfigq <awfigq@qq.com>
 */
class Validate {

    private $error = '';

    /**
     * 验证数据 
     * @param array $validate 验证设置
     * @param array $data 数据
     * @return bool
     */
    public static function validation($validate, $data) {
        $is_check = true;

        foreach ($validate as $val) {
            $key = $val[0];

            switch ($val[1]) {
                case 'min_length': // 验证字符串最小长度
                    $is_check = self::minLength($data[$key], $val[3]);
                    break;
                case 'max_length': // 验证字符串最大长度
                    $is_check = self::maxLength($data[$key], $val[3]);
                    break;
                case 'range_length': // 验证字符是否在某个长度范围内
                    $is_check = self::rangeLength($data[$key], $val[3], $val[4]);
                    break;
                case 'min': // 验证数字最小值
                    $is_check = self::min($data[$key], $val[3]);
                    break;
                case 'max': // 验证数字最大值
                    $is_check = self::max($data[$key], $val[3]);
                    break;
                case 'range': // 验证数字是否在某个大小范围内
                    $is_check = self::range($data[$key], $val[3], $val[4]);
                    break;
                case 'confirm': // 验证两个字段是否相同
                    $is_check = $data[$key] == $data[$val[3]];
                    break;
                case 'in': // 验证是否在某个数组范围之内
                    $is_check = in_array($data[$key], $val[3]);
                    break;
                case 'equal': // 验证是否等于某个值
                    $is_check = $data[$key] == $val[3];
                    break;
                case 'qq_msn': // 验证QQ或MSN是否合法
                    $is_check = self::qqMsn($data[$key]);
                    break;
                case 'regex':
                default:    // 默认使用正则验证 可以使用验证类中定义的验证名称
                    // 检查附加规则
                    $is_check = self::regex($data[$key], $val[1]);
                    break;
            }

            if (!$is_check) {
                $this->error = $val[2];
                break;
            }
        }

        return $is_check;
    }

    /**
     * 获取错误信息 
     * @return string
     */
    public static function getError() {
        return $this->error;
    }

    public static function required($value) {
        return self::regex($value, 'required');
    }

    public static function minLength($value, $length) {
        return $length <= getStrLen($value);
    }

    public static function maxLength($value, $length) {
        return $length >= getStrLen($value);
    }

    public static function rangeLength($value, $min_length, $max_length) {
        return $min_length <= getStrLen($value) && $max_length >= getStrLen($value);
    }

    public static function min($value, $num) {
        return $num <= $value;
    }

    public static function max($value, $num) {
        return $num >= $value;
    }

    public static function range($value, $min_num, $max_num) {
        return $min_num <= $value && $max_num >= $value;
    }

    public static function email($value) {
        return self::regex($value, 'email');
    }

    public static function url($value) {
        return self::regex($value, 'url');
    }

    public static function date($value) {
        return self::regex($value, 'date');
    }

    public static function currency($value) {
        return self::regex($value, 'currency');
    }

    public static function digits($value) {
        return self::regex($value, 'digits');
    }

    public static function double($value) {
        return self::regex($value, 'double');
    }

    public static function number($value) {
        return self::regex($value, 'number');
    }

    public static function equal($value, $value1) {
        return $value == $value1;
    }

    public static function qqMsn($value) {
        return self::regex($value, 'digits') || self::regex($value, 'email');
    }

    public static function english($value) {
        return self::regex($value, 'english');
    }

    public static function zipCode($value) {
        return self::regex($value, 'zipcode');
    }

    public static function regex($value, $rule) {
        $validate = array(
            'required' => '/.+/',
            'email' => "/^\w+((-\w+)|(\.\w+))*\@[A-Za-z0-9]+((\.|-)[A-Za-z0-9]+)*\.[A-Za-z0-9]+$/i",
            'url' => "/^http:\/\/[A-Za-z0-9]+\.[A-Za-z0-9]+[\/=\?%\-&_~`@[\]\':+!]*([^<>\"\"])*$/i",
            'date' => '/^\d{4}[\/-]\d{1,2}[\/-]\d{1,2}(?:|\s\d{1,2}:\d{1,2}(?:|:\d{1,2}))$/',
            'currency' => '/^\d+(\.\d+)?$/',
            'digits' => '/^\d+$/',
            'number' => '/^-?(?:\d+|\d{1,3}(?:,\d{3})+)(?:\.\d+)?$/',
            'zipcode' => '/^[0-9]\d{5}$/',
            'integer' => '/^[-\+]?\d+$/',
            'double' => '/^[-\+]?\d+(\.\d+)?$/',
            'english' => '/^[A-Za-z]+$/',
        );

        // 检查是否有内置的正则表达式
        if (isset($validate[strtolower($rule)])) {
            $rule = $validate[strtolower($rule)];
        }
        return preg_match($rule, $value) === 1;
    }

    /**
     * 验证手机号码
     * @param type $value
     * @return type
     * @author bant 2012-11-2 14:37:31 
     */
    public static function mobile($value) {
        return self::regex($value, "/^1[0-9]{10}$/");
    }

    /**
     * 验证手机+固话
     * @param type $value
     * @return type
     */
    public static function mobileAndphone($value) {
        return self::regex($value, '/^(0?1[0-9]\d{9})$|^((0(10|2[1-3]|[3-9]\d{2}))-?[1-9]\d{6,7})$/');
    }

}

?>