<?php

/**
 * 评论Service
 * @author     bant
 * @copyright  (C) 2011-2013 BPICMS
 * @license    http://www.chinabpi.com/license/
 * @version    1.0
 * @createTime 2013-4-18
 * @updateTime 2013-4-18
 */
class CommentService extends BaseService {

    protected $setting, $sideTableId, $commentModel, $isInAdmin;

    function __construct() {
        parent::__construct();
        $this->setting = BaseAction::getVar('config');
        $this->sideTableId = intval($this->setting['comment_default_side_table_id']); //评论存储分表
        $this->commentModel = D('Comment');
        $this->isInAdmin = defined('IN_ADMIN') && IN_ADMIN;   //是否后台
    }

    /**
     * 新增评论 / 回复评论
     * @param type $contentId       文章ID
     * @param type $content         评论内容    (数据必须是被转义过的)
     * @param type $data            表单数据
     * @param type $userId          用户ID      (后台传入值为0)
     * @param type $pid             评论父ID
     * @return boolean
     */
    public function add($contentId, $content, $data = array(), $userId = 0, $pid = 0) {
        $contentId = intval($contentId);
        $userId = intval($userId);
        $pid = intval($pid);

        $message = $pid == 0 ? '新增评论' : '回复评论';
        if ($this->isInAdmin) {
            $status = 1;
        } else {
            //后台可以设置游客评论
//            if (empty($userId)) {
//                $this->setError('用户ID不能为空');
//                return FALSE;
//            }
            //TODO: 是否需要抽出常量
            $status = ($this->setting['comment_is_check'] == 1) ? 0 : 1;
        }

        $commentData = array(
            'content_id' => $contentId,
            'author_ip' => get_client_ip(),
            'date' => time(),
            'status' => $status,
            'agent' => substr($_SERVER['HTTP_USER_AGENT'], 0, 254),
            'pid' => $pid,
            'user_id' => $userId,
            'side_table_id' => $this->sideTableId,
        );

        //评论设置检查
        $content = $this->_checkArg($content, $data, $userId, $pid);
        if (!$content) {
            return false;
        } else if (is_array($content)) {
            $commentData['status'] = 0;
            $content = $content['content'];
        }

        $commentId = $this->commentModel->add($commentData);
        if (!$commentId) {
            $this->setError($message . '失败');
            return false;
        }

        $commentExtendData = array(
            'comment_id' => $commentId,
            'content_id' => $contentId,
            'user_id' => $userId,
            'content' => $content,
            'version' => (string) $data['version']
        );
        if (!M('CommentData_' . $this->sideTableId)->add($commentExtendData)) {
            $this->commentModel->where(array('comment_id' => $commentId))->delete();
            $this->setError($message . '失败');
            return false;
        }

        if (!$this->isInAdmin) {
            session('comment_time', time());
        }

        return true;
    }

    /**
     * 编辑评论
     * @param type $commentId 评论ID
     * @param type $content   评论内容  (数据必须是被转义过的)
     * @param type $data      附加数据
     * @return boolean
     */
    public function edit($commentId, $content, $data = array()) {
//        if (!$this->isInAdmin) {
//            $this->setError('内部错误');
//            return false;
//        }

        $commentId = intval($commentId);
        $commentInfo = $this->getCommentById($commentId);
        if (empty($commentInfo)) {
            $this->setError('该评论不存在');
            return false;
        }

        // 用户判断
        if (!$this->isInAdmin && intval($commentInfo['user_id']) !== intval(BaseAction::getVar('user_id'))) {
            $this->setError('编辑评论失败');
            return FALSE;
        }

        //内容验证及过滤
        $content = $this->_checkArg($content, $data);
        if (!$content) {
            return false;
        } else if (is_array($content)) {
            $content = $content['content'];
        }

        $map = array('comment_id' => $commentId);
        // 副表数据
        $commentExtendData = array(
            'content' => $content,
        );

        $status = D('CommentData_' . $commentInfo['side_table_id'])->where($map)->save($commentExtendData);
        if (FALSE === $status) {
            $this->setError('评论编辑失败');
            return false;
        }
        return true;
    }

    /**
     * 删除评论
     * @param type $commentId   评论ID
     * @return boolean
     */
    public function delete($commentId, $userId = 0) {
        $userId = intval($userId);
        $commentId = intval($commentId);

        $count = 0;
        $commentInfo = $this->getCommentById($commentId);
        if ($commentInfo) {
            $map = array('comment_id' => $commentId);
            if (!$this->isInAdmin && intval($commentInfo['user_id']) !== $userId) {
                $this->setError('不能删除别人的评论');
                return FALSE;
            }

            $count = $this->commentModel->where($map)->delete();
            M('CommentData_' . $commentInfo['side_table_id'])->where($map)->delete();
        }
        return $count;
    }

    /**
     * 根据应用ID删除评论
     * @param type $contentId       文章ID
     * @return boolean              返回是否删除成功
     */
    public function deleteByContentId($contentId, $userId = 0) {
        $userId = intval($userId);
        $contentId = intval($contentId);
        if (!$contentId) {
            $this->setError('应用ID为空');
            return FALSE;
        }

        $map = array('content_id' => $contentId);
        if (!$this->isInAdmin) {
            $map['user_id'] = $userId;
        }

        // 删除主表
        $count = $this->commentModel->where($map)->delete();

        // 删除副表
        $index = 0;
        $commentTableName = 'comment_data_' . $index;
        while (D('Common')->isTableExists($commentTableName)) {
            M($commentTableName)->where($map)->delete();

            $index++;
            $commentTableName = 'comment_data_' . $index;
        }
        return $count;
    }

    /**
     * 根据用户id删除评论
     * @param type $userId      用户ID
     * @return boolean
     */
    public function deleteByUserId($userId) {
        $userId = intval($userId);
        if (!$userId) {
            $this->setError('用户ID为空');
            return FALSE;
        }

        $map = array('user_id' => $userId);
        // 删除主表
        $count = $this->commentModel->where($map)->delete();

        // 删除副表
        $index = 0;
        $commentTableName = 'comment_data_' . $index;
        while (D('Common')->isTableExists($commentTableName)) {
            M($commentTableName)->where($map)->delete();

            $index++;
            $commentTableName = 'comment_data_' . $index;
        }
        return $count;
    }

    /**
     * 根据评论ID获取评论详情
     * @staticvar array $commentList
     * @param type $commentId       评论ID
     * @param type $isOnlyPublish   是否需要通过审核
     * @return type
     */
    public function getCommentById($commentId, $isOnlyPublish = false) {
        static $commentList = array();
        $commentId = intval($commentId);
        if (!isset($commentList[$commentId])) {
            $map = array('comment_id' => $commentId);

            $commentList[$commentId] = $this->commentModel->where($map)->find();
            if ($commentList[$commentId]) {
                $extend = M('CommentData_' . $commentList[$commentId]['side_table_id'])->where($map)->find();
                if (!empty($extend)) {
                    $commentList[$commentId] = array_merge($extend, $commentList[$commentId]);
                }
            }
        }

        $commentInfo = $commentList[$commentId];
        if (is_array($commentInfo)) {
            // $isOnlyPublish只获取审核通过的评论
            if ($isOnlyPublish && !$commentInfo['status']) {
                return NULL;
            } else {
                return $commentInfo;
            }
        }
        return NULL;
    }

    /**
     * 根据应用ID获取评论列表
     * @param type $contentId                   文章ID 
     * @param type $pageIndex                   页码
     * @param type $listRow                     每页数据条数
     * @param type $isOnlyPublish               是否只获取通过审核的评论
     * @return type
     */
    public function getCommentListByContentId($contentId, $pageIndex = 1, $listRow = 10, $isOnlyPublish = false) {
        $contentId = intval($contentId);

        $map = array(
            'content_id' => $contentId
        );
        if ($isOnlyPublish) {
            $map['status'] = 1;
        }
        $count = $this->commentModel->field('comment_id')->where($map)->count();
        $page = page($count, $listRow, $pageIndex);

        // 拉取评论列表
        $commentIdList = $this->commentModel->field('comment_id')->where($map)->order('comment_id DESC')->limit($page->firstRow, $page->listRows)->select();
        $commentList = array();
        foreach ($commentIdList as $commentInfo) {
            $commentInfo = $this->getCommentById($commentInfo['comment_id'], $isOnlyPublish);
            if ($commentInfo) {
                $commentList[] = $this->_formatNormalComment($commentInfo);
            }
        }
        $returnData = array(
            'commentList' => $commentList,
            'page' => $page,
        );
        return $returnData;
    }

    /**
     * 根据内容id获取评论数
     * @param type $contentId
     * @param type $isOnlyPublish
     * @return boolean
     */
    public function getCommentCountByContentId($contentId, $isOnlyPublish = true) {
        $contentId = intval($contentId);
        if (!$contentId) {
            $this->setError('参数缺失');
            return false;
        }
        $map = array(
            'content_id' => $contentId
        );
        if ($isOnlyPublish) {
            $map['status'] = 1;
        }
        return $this->commentModel->field('comment_id')->where($map)->count();
    }

    /**
     * 格式化一般程度的评论
     * @param type $originComment
     * @return type
     */
    private function _formatNormalComment($originComment) {
        if ($originComment['user_id']) {
            $userName = getUserNameById($originComment['user_id']);
        } else {
            $userName = hideTheLastOneIp($originComment['author_ip']);
        }
        return array(
            'comment_id' => $originComment['comment_id'],
            'user_id' => $originComment['user_id'],
            'user_name' => $userName,
            'content' => $originComment['content'],
            'date' => date('Y-m-d H:i:s', $originComment['date']),
        );
    }

    /**
     * 检测评论参数/字段是否正确
     * @param type $content         评论内容
     * @param type $data            评论字段
     * @param type $userId          用户ID
     * @param type $pid             评论父ID
     * @return type                 成功返回处理后的$content，失败返回false
     */
    private function _checkArg($content, $data = array(), $userId = 0, $pid = 0) {
        //是否开启评论
        $setting = $this->setting;
        if (!$setting['comment_status']) {
            $this->setError('站点评论被关闭');
            return false;
        }

        //是否允许游客评论
        if (!$this->isInAdmin && !$setting['comment_is_allow_guest'] && $userId == 0) {
            $this->setError('请先登录');
            return false;
        }

        //评论时间检查
        $commentInterval = intval($setting['comment_interval']);
        if (!$this->isInAdmin && $commentInterval > 0 && time() - session('comment_time') < $commentInterval) {
            $this->setError('两次评论的间隔不能少于 ' . $commentInterval . ' 秒');
            return false;
        }

        //评论最大字节数
        if (empty($content)) {
            $this->setError('评论内容不能为空');
            return false;
        }

        $commentMaxLength = intval($setting['comment_max_length']);
        if ($commentMaxLength > 0 && strlen($content) > $commentMaxLength) {
            $this->setError('评论超出最大长度');
            return false;
        }

        //评论父对象是否存在
        if (!$this->isInAdmin && $pid && !$this->getCommentById($pid, true)) {
            $this->setError('回复的评论不存在');
            return false;
        }
        //过滤评论内容
        $content = BService('Filter')->filterContent($content);
        if (!$content) {
            $this->setError(BService('Filter')->getError());
            return false;
        }
        return $content;
    }

}

?>
