<?php

/**
 * 市场视图service
 * @author     bant
 * @copyright  (C) 2011-2013 BPICMS
 * @license    http://www.chinabpi.com/license/
 * @version    1.0
 * @createTime 2013-11-11
 * @updateTime 2013-11-11
 */
class MarketViewService extends BaseService {

    const APPLICATION_ORDER_ENUM_LATEST = 'LATEST'; //程序排序枚举-最新
    const FOTMAT_TYPE_ENUM_SIMPLE = 'SIMPLE'; //格式化输出程度枚举-精简
    const FOTMAT_TYPE_ENUM_SIMPLE_WITH_THUMB = 'SIMPLE_THUMB'; //格式化输出程度枚举-精简&缩略图
    const FOTMAT_TYPE_ENUM_NORMAL = 'NORMAL'; //格式化输出程度枚举-一般
    const FOTMAT_TYPE_ENUM_DETAIL = 'DETAIL'; //格式化输出程度枚举-详细

    public function __construct() {
        parent::__construct();
    }

    /**
     * 获取最新app版本下载链接
     * @param type $cacheTime
     * @return type
     */
    public function getLastestAppVersionUrl($cacheTime = 86400) {
        $cacheName = 'ViewLastestAppVersionUrl';
        if ($cacheTime && $cache = S($cacheName)) {
            return $cache;
        }

        $appList = array();
        $androidVersion[] = BService('AppVersion', 'AppVersion')->getLastAppVersion(C('APP_VERSION_TYPE.ANDROID'));
        $padVersion[] = BService('AppVersion', 'AppVersion')->getLastAppVersion(C('APP_VERSION_TYPE.ANDROID_PAD'));
        $dataList = array_merge((array) $androidVersion, (array) $padVersion);

        foreach ($dataList as $_data) {
            $appList[$_data['type']] = getAttachmentSiteUrl($_data['url']);
        }

        if ($cacheTime) {
            S($cacheName, $appList, $cacheTime);
        }
        return $appList;
    }

    /**
     * 根据topic flag获取程序列表
     * @param type $positionFlag
     * @param type $limitCount
     * @param type $cacheTime
     * @return type
     */
    public function getApplicationListByTopicFlag($positionFlag, $limitCount = 5, $cacheTime = 86400) {
//        return $this->getApplicationListByPositionId(null, $positionFlag, $limitCount, self::FOTMAT_TYPE_ENUM_SIMPLE_WITH_THUMB, $cacheTime);
    }

    /**
     * 获取首页顶部轮播数据
     * @param type $limitCount
     * @param type $cacheTime
     * @return type
     */
    public function getTopSlider($limitCount = 5, $cacheTime = 86400) {
        return $this->getApplicationListByPositionId(null, C('POSITION_FLAG.TOP_SLIDER'), $limitCount, self::FOTMAT_TYPE_ENUM_SIMPLE_WITH_THUMB, $cacheTime);
    }

    /**
     * 获取首页特别推荐数据
     * @param type $limitCount
     * @param type $cacheTime
     * @return type
     */
    public function getSpecialRecommended($limitCount = 5, $cacheTime = 86400) {
        return $this->getApplicationListByPositionId(null, C('POSITION_FLAG.SPECIAL_RECOMMENDED'), $limitCount, self::FOTMAT_TYPE_ENUM_SIMPLE_WITH_THUMB, $cacheTime);
    }

    /**
     * 获取推荐应用列表
     * @param type $limitCount              返回数量
     * @param type $isCache                 是否读写缓存
     * @param type $cacheTime               缓存时间
     * @return type
     */
    public function getRecommendedAppList($limitCount = 20, $cacheTime = 86400) {
        return $result = $this->getApplicationListByPositionId(null, C('POSITION_FLAG.RECOMMENDED_APPLICATION'), $limitCount, self::FOTMAT_TYPE_ENUM_SIMPLE, $cacheTime);
    }

    /**
     * 获取每天一荐程序
     * @param type $limitCount              返回数量
     * @param type $isCache                 是否读写缓存
     * @param type $cacheTime               缓存时间
     * @return type
     */
    public function getOneDayOneApplication($cacheTime = 86400) {
        $result = $this->getApplicationListByPositionId(null, C('POSITION_FLAG.ONE_DAY_ONE_APPLICATION'), 1, self::FOTMAT_TYPE_ENUM_NORMAL, $cacheTime);
        return $result;
    }

    /**
     * 根据positionid获取程序列表
     * @param type $positionId
     * @param type $flag
     * @param type $limitCount
     * @param type $cacheTime
     * @return boolean
     */
    public function getApplicationListByPositionId($positionId = NULL, $flag = NULL, $limitCount = 0, $formatType = self::FOTMAT_TYPE_ENUM_SIMPLE, $cacheTime = 86400) {
        if (!($positionId || $flag)) {
            $this->setError('参数缺失');
            return false;
        }

        $cacheName = 'ViewApplicationListByPosition' . 'Id' . $positionId . 'F' . $flag . 'N' . $limitCount;
        if ($cacheTime && $cache = S($cacheName)) {
            return $cache;
        }

        $result = BService('Position')->getPositionDataList($positionId, $flag, 1);
        $positionDataList = $result['positionList'];
        $i = 0;
        $appList = array();
        foreach ($positionDataList as $_positionData) {
            $applicationData = BService('Application', 'Market')->getApplicationById($_positionData['content_id'], NULL, NULL, false, TRUE);
            //获取推荐位缩略图
            $applicationData['thumb'] = $_positionData['data']['thumb'];
            if ($applicationData) {
                if ($formatType == self::FOTMAT_TYPE_ENUM_SIMPLE) {
                    $appList[] = $this->_formatSimpleApplicaition($applicationData);
                } elseif ($formatType == self::FOTMAT_TYPE_ENUM_SIMPLE_WITH_THUMB) {
                    $appList[] = $this->_formatSimpleWithThumbApplicaition($applicationData);
                } else {
                    $appList[] = $this->_formatNormalApplicaition($applicationData);
                }
                if ($limitCount) {//限制返回条数处理
                    $i++;
                    if ($i >= $limitCount) {
                        break;
                    }
                }
            }
        }
        if ($cacheTime) {
            S($cacheName, $appList, $cacheTime);
        }
        return $appList;
    }

    /**
     * 获取程序列表
     * @param type $pageIndex
     * @param type $listRow
     * @param type $formatType
     * @param type $order
     * @param type $cacheTime
     * @return type
     */
    public function getApplicationList($categoryId = null, $modelName = NULL, $pageIndex = 1, $listRow = 20, $formatType = self::FOTMAT_TYPE_ENUM_SIMPLE, $order = 'id DESC', $cacheTime = 86400) {
        $applicationList = array();
        $cacheName = 'ViewLatestApplicationList' . 'C' . $categoryId . 'MN' . $modelName . 'P' . $pageIndex . 'L' . $listRow . $formatType;

        if ($cacheTime && $applicationListCache = S($cacheName)) {
            return $applicationListCache;
        }

        if ($modelName) {
            $modelId = D('Model')->getModelIdByModelName($modelName, C('MODEL_TYPE.APPLICATION'));
        }

        $result = BService('Application', 'Market')->getContentList($categoryId, $modelId, $pageIndex, $listRow, $order, $cacheTime);
        if ($result) {
            foreach ($result['contentList'] as $_application) {
                switch ($formatType) {
                    case self::FOTMAT_TYPE_ENUM_SIMPLE:
                        $_application = $this->_formatSimpleApplicaition($_application);
                        break;
                    case self::FOTMAT_TYPE_ENUM_SIMPLE_WITH_THUMB:
                        $_application = $this->_formatSimpleWithThumbApplicaition($_application);
                        break;
                    case self::FOTMAT_TYPE_ENUM_DETAIL:
                        $_application = $this->_formatDetailApplication($_application);
                        break;
                    case self::FOTMAT_TYPE_ENUM_NORMAL:
                    default:
                        $_application = $this->_formatNormalApplicaition($_application);
                        break;
                }
                $applicationList[] = $_application;
            }
        }
        $cacheList = array(
            'applicationList' => $applicationList,
            'page' => $result['page'],
        );
        if ($cacheTime) {
            S($cacheName, $cacheList, $cacheTime);
        }
        return $cacheList;
    }

    /**
     * 获取最新程序列表
     * @param null $categoryId
     * @param null $modelName
     * @param type $pageIndex
     * @param type $listRow
     * @param type $formatType
     * @param type $cacheTime
     * @return type
     */
    public function getLatestApplicationList($categoryId = NULL, $modelName = null, $pageIndex = 1, $listRow = 20, $formatType = self::FOTMAT_TYPE_ENUM_SIMPLE, $cacheTime = 86400) {
        return $this->getApplicationList($categoryId, $modelName, $pageIndex, $listRow, $formatType, 'update_time DESC', $cacheTime);
    }

    /**
     * 获取评分排行榜程序列表
     * @param type $applicationModelName    模型名字
     * @param type $categoryId              栏目id    
     * @param type $pageIndex               页码
     * @param type $listRow                 每页条数
     * @param type $formatType              格式化类型
     * @param type $cacheTime
     * @return type
     */
    public function getScoreTopApplicationList($categoryId = NULL, $applicationModelName = NULL, $pageIndex = 1, $listRow = 20, $formatType = self::FOTMAT_TYPE_ENUM_SIMPLE, $hitOrderType = NULL, $cacheTime = 86400) {
        $applicationList = array();
        $cacheName = 'ViewScoreTop' . ucwords($applicationModelName) . 'Category' . intval($categoryId) . 'List' . 'P' . $pageIndex . 'L' . $listRow . $formatType;

        if ($cacheTime && $scoreTopApplicationListCache = S($cacheName)) {
            return $scoreTopApplicationListCache;
        }
        //获取程序类型
        $applicationType = D('Market/Application')->getApplicationTypeByModelName($applicationModelName);
        $result = BService('ApplicationComment', 'Market')->getScoreTopApplicationList($applicationType, $categoryId, $pageIndex, $listRow);
        $applicationIds = $result['applicationList'];
        $page = $result['page'];
        foreach ($applicationIds as $_application) {
            $_data = BService('Application', 'Market')->getApplicationById($_application['application_id'], NULL, NULL, FALSE, TRUE);
            if ($_data) {
                $_data['score'] = $_application['score'];
                if ($formatType == self::FOTMAT_TYPE_ENUM_SIMPLE) {
                    $applicationList[] = $this->_formatSimpleApplicaition($_data);
                } else {
                    $applicationList[] = $this->_formatNormalApplicaition($_data);
                }
            }
        }

        $cacheList = array(
            'applicationList' => $applicationList,
            'page' => $page,
        );
        if ($cacheTime) {
            S($cacheName, $cacheList, $cacheTime);
        }

        return $cacheList;
    }

    /**
     * 获取下载排行榜程序列表
     * @param type $categoryId              栏目id    
     * @param type $applicationModelName    模型名字
     * @param type $pageIndex               页码
     * @param type $listRow                 每页条数
     * @param type $formatType              格式化类型
     * @param type $cacheTime
     * @return type
     */
    public function getDownloadTopApplicationList($categoryId = NULL, $applicationModelName = NULL, $pageIndex = 1, $listRow = 20, $formatType = self::FOTMAT_TYPE_ENUM_SIMPLE, $hitOrderType = NULL, $cacheTime = 86400) {
        $applicationList = array();
        $cacheName = 'ViewDownloadTop' . ucwords($applicationModelName) . 'Category' . intval($categoryId) . 'List' . 'P' . $pageIndex . 'L' . $listRow . $formatType;

        if ($cacheTime && $downloadTopAppList = S($cacheName)) {
            return $downloadTopAppList;
        }
        //获取程序类型
        $applicationType = D('Market/Application')->getApplicationTypeByModelName($applicationModelName);
        $hitData = BService('Hit')->getHit(C('ACCESS_TYPE.APPLICATION'), NULL, $applicationType, C('HIT_FLAG.DOWNLOAD'), $categoryId, $listRow, $pageIndex, $hitOrderType);
        if ($hitData) {
            foreach ($hitData['hitList'] as $_app) {
                $applicationInfo = BService('Application', 'Market')->getApplicationById($_app['access_id'], NULL, NULL, true, true);
                if (empty($applicationInfo)) {
                    continue;
                }
                $applicationInfo['hit_count'] = $_app['total_count'];
                if ($applicationInfo) {
                    if ($formatType == self::FOTMAT_TYPE_ENUM_SIMPLE) {
                        $applicationList[] = $this->_formatSimpleApplicaition($applicationInfo);
                    } else {
                        $applicationList[] = $this->_formatNormalApplicaition($applicationInfo);
                    }
                }
            }
        }
        $cacheList = array(
            'applicationList' => $applicationList,
            'page' => $hitData['page'],
        );
        if ($cacheTime) {
            S($cacheName, $cacheList, $cacheTime);
        }
        return $cacheList;
    }

    /**
     * 获取程序栏目列表
     * @param type $modelTableName
     * @param type $isCache
     * @param type $cacheTime
     * @return boolean
     */
    public function getApplicationCategoryList($modelTableName, $cacheTime = 86400) {
        return $this->_getCategoryList('Application', $modelTableName, C('MODEL_TYPE.APPLICATION'), C('CATEGORY_TYPE.NORMAL'), $cacheTime);
    }

    /**
     * 根据栏目id获取专题信息
     * @param type $categoryId
     * @return boolean
     */
    public function getTopicByCategoryId($categoryId) {
        if (empty($categoryId)) {
            $this->setError('参数缺失');
            return false;
        }
        $categoryInfo = BService('Category')->getCategoryInfo($categoryId, C('CATEGORY_TYPE.TOPIC'));
        if ($categoryInfo) {
            $categoryInfo = $this->_formatCategoryInfo($categoryInfo);
        }
        return $categoryInfo;
    }

    /**
     * 获取专题栏目列表
     * @param type $cacheTime
     * @return type
     */
    public function getTopicCategoryList($cacheTime = 86400) {
        $topicList = $this->_getCategoryList('Topic', NULL, NULL, C('CATEGORY_TYPE.TOPIC'), $cacheTime);
        if ($topicList) {
            foreach ($topicList as $categoryId => $_topic) {
                $result = $this->getApplicationList($categoryId, null, 1, 4);
                $applicationList = $result['applicationList'];
                $_topic['applicationList'] = $applicationList;
                $topicList[$categoryId] = $_topic;
            }
        }
        return $topicList;
    }

    /**
     * 获取文章列表
     * @param type $modelName
     * @param type $pageIndex
     * @param type $listRow
     * @param type $formatType
     * @param type $order
     * @param type $cacheTime
     * @return boolean
     */
    public function getContentList($modelName, $pageIndex = 1, $listRow = 20, $formatType = self::FOTMAT_TYPE_ENUM_SIMPLE, $order = 'update_time DESC', $cacheTime = 86400) {
        if (empty($modelName)) {
            $this->setError('参数缺失');
            return false;
        }

        $contentList = array();
        $cacheName = 'ViewContentList' . 'MN' . $modelName . 'P' . $pageIndex . 'L' . $listRow . $formatType;

        if ($cacheTime && $contentCache = S($cacheName)) {
            return $contentCache;
        }

        $modelId = D('Model')->getModelIdByModelName($modelName, C('MODEL_TYPE.CONTENT'));

        $result = BService('Content')->getContentList(NULL, $modelId, $pageIndex, $listRow, $order, $cacheTime);

        if ($result) {
            foreach ($result['contentList'] as $_content) {
                if ($formatType == self::FOTMAT_TYPE_ENUM_SIMPLE) {
                    $contentList[] = $this->_formatSimpleContent($_content);
                } else {
                    $contentList[] = $this->_formatNormalContent($_content);
                }
            }
        }
        $cacheList = array(
            'contentList' => $contentList,
            'page' => $result['page'],
        );
        if ($cacheTime) {
            S($cacheName, $cacheList, $cacheTime);
        }

        return $cacheList;
    }

    /**
     * 格式化输出精简栏目列表
     * @param type $dataList
     * @return boolean
     */
    public function formatSimpleCategoryList($dataList) {
        if (empty($dataList)) {
            $this->setError('参数缺失');
            return false;
        }

        foreach ($dataList as $_category) {
            $categoryList[] = array(
                'category_id' => $_category['category_id'],
                'category_name' => $_category['category_name'],
                'image' => getAttachmentSiteUrl($_category['image']),
            );
        }
        return $categoryList;
    }

    /**
     * 格式化输出详细程序信息
     * @param type $originApplication
     * @return type
     */
    public function formatDetailApplication($originApplication) {
        return $this->_formatDetailApplication($originApplication);
    }

    /**
     * 格式化输出精简程序信息
     * @param type $originApplication
     * @return type
     */
    public function formatSimpleApplicaition($originApplication) {
        return $this->_formatSimpleApplicaition($originApplication);
    }

    /**
     * 格式化输出常用程序信息
     * @param type $originApplication
     * @return type
     */
    public function formatNormalApplicaition($originApplication) {
        return $this->_formatNormalApplicaition($originApplication);
    }

    /**
     * 格式化输出一般文章信息
     * @param type $originContent
     * @return type
     */
    public function formatNormalContent($originContent) {
        return $this->_formatNormalContent($originContent);
    }

    /**
     * 格式化输出详细文章信息
     * @param type $originContent
     * @return type
     */
    public function formatDetailContent($originContent) {
        return $this->_formatDetailContent($originContent);
    }

    /**
     * 格式化输出详细程序信息
     * @param type $originApplication
     * @return type
     */
    public function _formatDetailApplication($originApplication) {
        $originApplication = $this->_handleData('Application', $originApplication);
        $id = $originApplication['id'];
        $screenshotList = $this->_formatScreenshot($originApplication['screenshot']);
        $hitInfo = BService('Hit')->getHit(C('ACCESS_TYPE.APPLICATION'), $id, C('APPLICATION_TYPE.APP'), C('HIT_FLAG.DOWNLOAD'));
        $downloadAttachmentId = current(array_pop(unserialize($originApplication['download_url'])));
        return array(
            'id' => $id,
            'title' => $originApplication['title'],
            'icon' => getAttachmentSiteUrl($originApplication['icon']),
            'hit_count' => intval($hitInfo[$id]['total_count']),
            'post_time' => date('Y-m-d', $originApplication['post_time']),
            'url' => BService('Url')->getContentUrl($originApplication['category_id'], $id),
            'os_version' => sprintf("%1.1f", $originApplication['os_version'] / 10) . '以上',
            'download_url' => $this->_formatDownloadUrl($id),
            'qr_code_url' => getAttachmentQRCode($downloadAttachmentId),
            'description' => $originApplication['description'],
            'model_id' => $originApplication['model_id'],
            'update_time' => date('Y-m-d', $originApplication['update_time']),
            'author' => $originApplication['author'],
            'version' => $originApplication['version'],
            'score' => ceil(isset($originApplication['score']) ? ($originApplication['score']) : BService('ApplicationComment', 'Market')->getScoreByApplicationId($id)),
            'version_code' => $originApplication['version_code'],
            'package_name' => $originApplication['package_name'],
            'screenshot' => $screenshotList,
            'size' => formatSize($originApplication['size']),
            'content' => bpiHtmlspecialcharsDecode($originApplication['content']),
        );
    }

    /**
     * 格式化输出简要+缩略图程序信息
     * @param type $originApplication
     * @return type
     */
    private function _formatSimpleWithThumbApplicaition($originApplication) {
        return $this->_formatSimpleApplicaition($originApplication, true);
    }

    /**
     * 格式化输出简要程序信息
     * @param type $originApplication
     * @return type 
     */
    private function _formatSimpleApplicaition($originApplication, $isWithThumb = false) {
        $originApplication = $this->_handleData('Application', $originApplication);
        $id = $originApplication['id'];
        $data = array(
            'id' => $id,
            'title' => $originApplication['title'],
            'icon' => getAttachmentSiteUrl($originApplication['icon']),
            'url' => BService('Url')->getContentUrl($originApplication['category_id'], $id),
            'hit_count' => intval($originApplication['hit_count']),
            'download_url' => $this->_formatDownloadUrl($id),
            'score' => ceil(isset($originApplication['score']) ? $originApplication['score'] : BService('ApplicationComment', 'Market')->getScoreByApplicationId($id)),
            'version_code' => $originApplication['version_code'],
            'package_name' => $originApplication['package_name'],
        );
        if ($isWithThumb && $originApplication['thumb']) {
            $data['thumb'] = getAttachmentSiteUrl($originApplication['thumb']);
        }
        return $data;
    }

    /**
     * 格式化输出一般程序信息
     * @param type $originApplication
     * @return type 
     */
    private function _formatNormalApplicaition($originApplication) {
        $originApplication = $this->_handleData('Application', $originApplication);
        $id = $originApplication['id'];
        $screenshotList = $this->_formatScreenshot($originApplication['screenshot']);
        $hitInfo = BService('Hit')->getHit(C('ACCESS_TYPE.APPLICATION'), $id, C('APPLICATION_TYPE.APP'), C('HIT_FLAG.DOWNLOAD'));
        return array(
            'id' => $id,
            'title' => $originApplication['title'],
            'icon' => getAttachmentSiteUrl($originApplication['icon']),
            'hit_count' => intval($hitInfo[$id]['total_count']),
            'post_time' => date('Y-m-d', $originApplication['post_time']),
            'url' => BService('Url')->getContentUrl($originApplication['category_id'], $id),
            'os_version' => sprintf("%1.1f", $originApplication['os_version'] / 10) . '以上',
            'download_url' => $this->_formatDownloadUrl($id),
            'description' => cutString($originApplication['description'], 0, 100),
            'score' => ceil(isset($originApplication['score']) ? $originApplication['score'] : BService('ApplicationComment', 'Market')->getScoreByApplicationId($id)),
            'screenshot' => $screenshotList,
            'size' => formatSize($originApplication['size']),
            'version' => $originApplication['version'],
            'update_time' => date('Y-m-d', $originApplication['update_time']),
        );
    }

    /**
     * 格式化输出简要文章信息
     * @param type $originContent
     * @return type 
     */
    private function _formatSimpleContent($originContent) {
        $originContent = $this->_handleData('Content', $originContent);
        $id = $originContent['id'];
        return array(
            'id' => $id,
            'title' => $originContent['title'],
            'url' => BService('Url')->getContentUrl($originContent['category_id'], $id),
        );
    }

    /**
     * 格式化输出一般文章信息
     * @param type $originContent
     * @return type 
     */
    private function _formatNormalContent($originContent) {
        $originContent = $this->_handleData('Content', $originContent);
        $id = $originContent['id'];
        return array(
            'id' => $id,
            'title' => $originContent['title'],
            'url' => $originContent['url'],
            'description' => $originContent['description'],
            'update_time' => $originContent['update_time'],
        );
    }

    /**
     * 格式化输出详细文章信息
     * @param type $originContent
     * @return type
     */
    private function _formatDetailContent($originContent) {
        $originContent = $this->_handleData('Content', $originContent);
        $id = $originContent['id'];
        //浏览数
        $hitData = BService('Hit')->getHit(C('ACCESS_TYPE.CONTENT'), $id, NULL, C('HIT_FLAG.HIT'), $originContent['category_id']);
        if ($hitData) {
            $hitData = current($hitData);
            $hitCount = $hitData['total_count'];
        }
        return array(
            'id' => $id,
            'title' => $originContent['title'],
            'url' => BService('Url')->getContentUrl($originContent['category_id'], $id),
            'description' => $originContent['description'],
            'update_time' => date('Y-m-d', $originContent['update_time']),
            'user_name' => getUserNameById($originContent['user_id'], $originContent['is_admin_add']),
            'from' => $originContent['from'],
            'comment_count' => BService('Comment', 'Comment')->getCommentCountByContentId($id),
            'hit_count' => $hitCount,
            'content' => bpiHtmlspecialcharsDecode($originContent['content']),
        );
    }

    /**
     * 通用数据处理
     * @param type $data     数据处理
     * @return type
     */
    private function _handleData($mainTableName, $data) {
        if ($data['pid']) {
            $relationModel = new AutoRelationModel($mainTableName);
            return $relationModel->relation(TRUE)->find($data['pid']);
        } else {
            return $data;
        }
    }

    /**
     * 格式化栏目信息
     * @param array $originCategoryInfo
     */
    private function _formatCategoryInfo($originCategoryInfo) {
        $originCategoryInfo['image'] = getAttachmentSiteUrl($originCategoryInfo['image']);
        return $originCategoryInfo;
    }

    /**
     * 获得栏目列表
     * @param string $cacheName
     * @param type $modelTableName
     * @param type $modelType
     * @param type $cacheTime
     * @return boolean
     */
    private function _getCategoryList($cacheName, $modelTableName = NULL, $modelType = NULL, $categoryType = NULL, $cacheTime = 86400) {
        if ($modelType && empty($modelTableName)) {
            $this->setError('参数缺失');
            return false;
        }
        $modelId = NULL;
        $cacheName = 'View' . $cacheName . ucwords($modelTableName) . 'CategoryList';
        if ($cacheTime && $appCategoryListCache = S($cacheName)) {
            return $appCategoryListCache;
        }
        if ($modelType) {
            $modelId = D('Model')->getModelIdByModelName($modelTableName, $modelType);
        }

        $categoryList = BService('Category')->getCategoryTreeList($modelId, false, 'is_set_html', true, true, $categoryType);
        // 加工一下数据，只抽取子栏目
        //TODO:当二级分类有三级分类的时候，二级分类也被过滤
        if ($categoryType == C('CATEGORY_TYPE.NORMAL')) {
            foreach ($categoryList as $key => $category) {
                if (empty($category['pid_array'])) {
                    unset($categoryList[$key]);
                }
            }
        }
        foreach ($categoryList as $key => $_category) {
            $categoryList[$key] = $this->_formatCategoryInfo($_category);
        }
        if ($cacheTime) {
            S($cacheName, $categoryList, $cacheTime);
        }
        return $categoryList;
    }

    /**
     * 格式化截图字段
     * @param type $originScreenshotList
     * @return type
     */
    private function _formatScreenshot($originScreenshotList) {
        $screenshotList = array();
        if ($originScreenshotList) {
            $originScreenshotList = unserialize($originScreenshotList);
            foreach ($originScreenshotList as $_originScreenshot) {
                $screenshotList[] = getAttachmentUrl($_originScreenshot['attachment_id']);
            }
        }
        return $screenshotList;
    }

    /**
     * 格式化下载url
     * @param type $originDownloadUrl
     * @return type
     */
    private function _formatDownloadUrl($applicationId) {
        return getApplicationDownloadUrl($applicationId);
    }

}

?>
