<?php

/**
 * 评论Service
 * @author     bant
 * @copyright  (C) 2011-2013 BPICMS
 * @license    http://www.chinabpi.com/license/
 * @version    1.0
 * @createTime 2013-4-18
 * @updateTime 2013-4-18
 */
class ApplicationCommentService extends BaseService {

    protected $setting, $sideTableId, $commentModel, $isInAdmin;

    function __construct() {
        parent::__construct();
        $this->setting = BaseAction::getVar('config');
        $this->sideTableId = intval($this->setting['application_comment_default_side_table_id']); //评论存储分表
        $this->commentModel = D('Market/ApplicationComment');
        $this->isInAdmin = defined('IN_ADMIN') && IN_ADMIN;   //是否后台
    }

    /**
     * 新增评论 / 回复评论
     * @param type $applicationId   应用ID
     * @param type $content         评论内容    (数据必须是被转义过的)
     * @param type $data            表单数据
     * @param type $userId          用户ID      (后台传入值为0)
     * @param type $pid             评论父ID
     * @return boolean
     */
    public function add($applicationId, $content, $data = array(), $userId = 0, $pid = 0) {
        $applicationId = intval($applicationId);
        $userId = intval($userId);
        $pid = intval($pid);

        // 检查应用是否存在
        $applicationInfo = D('Application')->field('category_id,application_type')->where(array('id' => $applicationId))->find();
        if (empty($applicationInfo)) {
            $this->setError('内部错误');
            return FALSE;
        }

        $categoryId = $applicationInfo['category_id'];
        $applicationType = $applicationInfo['application_type'];
        $message = $pid == 0 ? '新增评论' : '回复评论';
        if ($this->isInAdmin) {
            $status = 1;
        } else {
            //后台可以设置游客评论
//            if (empty($userId)) {
//                $this->setError('用户ID不能为空');
//                return FALSE;
//            }
            //TODO: 是否需要抽出常量
            $status = ($this->setting['application_comment_is_check'] == 1) ? 0 : 1;
        }

        $commentData = array(
            'application_id' => $applicationId,
            'category_id' => $categoryId,
            'author_ip' => get_client_ip(),
            'date' => time(),
            'status' => $status,
            'agent' => substr($_SERVER['HTTP_USER_AGENT'], 0, 254),
            'pid' => $pid,
            'user_id' => $userId,
            'side_table_id' => $this->sideTableId,
            'score' => intval($data['score']),
            'flag' => $applicationType,
        );

        //评论设置检查
        $content = $this->_checkArg($content, $data, $userId, $pid);
        if (!$content) {
            return false;
        } else if (is_array($content)) {
            $commentData['status'] = 0;
            $content = $content['content'];
        }

        $commentId = $this->commentModel->add($commentData);
        if (!$commentId) {
            $this->setError($message . '失败');
            return false;
        }

        $commentExtendData = array(
            'comment_id' => $commentId,
            'application_id' => $applicationId,
            'user_id' => $userId,
            'content' => $content,
            'version' => (string) $data['version']
        );
        if (!M('ApplicationCommentData_' . $this->sideTableId)->add($commentExtendData)) {
            $this->commentModel->where(array('comment_id' => $commentId))->delete();
            $this->setError($message . '失败');
            return false;
        }

        if (!$this->isInAdmin) {
            session('comment_time', time());
        }

        return true;
    }

    /**
     * 编辑评论
     * @param type $commentId 评论ID
     * @param type $content   评论内容  (数据必须是被转义过的)
     * @param type $data      附加数据
     * @return boolean
     */
    public function edit($commentId, $content, $data = array()) {
//        if (!$this->isInAdmin) {
//            $this->setError('内部错误');
//            return false;
//        }

        $commentId = intval($commentId);
        $commentInfo = $this->getCommentById($commentId);
        if (empty($commentInfo)) {
            $this->setError('该评论不存在');
            return false;
        }

        // 用户判断
        if (!$this->isInAdmin && intval($commentInfo['user_id']) !== intval(BaseAction::getVar('user_id'))) {
            $this->setError('编辑评论失败');
            return FALSE;
        }

        //内容验证及过滤
        $content = $this->_checkArg($content, $data);
        if (!$content) {
            return false;
        } else if (is_array($content)) {
            $content = $content['content'];
        }

        $map = array('comment_id' => $commentId);
        // 主表数据
        $commentData = array(
            'score' => intval($data['score'])
        );

        if (FALSE === $this->commentModel->where($map)->save($commentData)) {
            $this->setError('评论编辑失败');
            return FALSE;
        }


        // 副表数据
        $commentExtendData = array(
            'content' => $content,
            'version' => $data['version']
        );

        $status = D('ApplicationCommentData_' . $commentInfo['side_table_id'])->where($map)->save($commentExtendData);
        if (FALSE === $status) {
            $this->setError('评论编辑失败');
            return false;
        }
        return true;
    }

    /**
     * 删除评论
     * @param type $commentId   评论ID
     * @return boolean
     */
    public function delete($commentId, $userId = 0) {
        $userId = intval($userId);
        $commentId = intval($commentId);
        if (!$commentId) {
            $this->setError('参数缺失');
            return false;
        }
        $count = 0;
        $commentInfo = $this->getCommentById($commentId);
        if ($commentInfo) {
            $map = array('comment_id' => $commentId);
            if (!$this->isInAdmin && intval($commentInfo['user_id']) !== $userId) {
                $this->setError('不能删除别人的评论');
                return FALSE;
            }

            $count = $this->commentModel->where($map)->delete();
            M('ApplicationCommentData_' . $commentInfo['side_table_id'])->where($map)->delete();
        }
        return $count;
    }

    /**
     * 根据应用ID删除评论
     * @param type $applicationId   应用ID
     * @return boolean              返回是否删除成功
     */
    public function deleteByApplicationId($applicationId, $userId = 0) {
        $userId = intval($userId);
        $applicationId = intval($applicationId);
        if (!$applicationId) {
            $this->setError('参数缺失');
            return false;
        }
        $map = array('application_id' => $applicationId);
        if (!$this->isInAdmin) {
            $map['user_id'] = $userId;
        }

        // 删除主表
        $count = $this->commentModel->where($map)->delete();

        // 删除副表
        $index = 0;
        $commentTableName = 'application_comment_data_' . $index;
        while (D('Common')->isTableExists($commentTableName)) {
            M($commentTableName)->where($map)->delete();

            $index++;
            $commentTableName = 'application_comment_data_' . $index;
        }
        return $count;
    }

    /**
     * 根据用户id删除评论
     * @param type $userId      用户ID
     * @return boolean
     */
    public function deleteByUserId($userId) {
        $userId = intval($userId);
        if (!$userId) {
            $this->setError('用户ID为空');
            return FALSE;
        }

        $map = array('user_id' => $userId);
        // 删除主表
        $count = $this->commentModel->where($map)->delete();

        // 删除副表
        $index = 0;
        $commentTableName = 'application_comment_data_' . $index;
        while ($this->isTableExists($commentTableName)) {
            M($commentTableName)->where($map)->delete();

            $index++;
            $commentTableName = 'application_comment_data_' . $index;
        }
        return $count;
    }

    /**
     * 根据评论ID获取评论详情
     * @staticvar array $commentList
     * @param type $commentId       评论ID
     * @param type $isOnlyPublish   是否需要通过审核
     * @return type
     */
    public function getCommentById($commentId, $isOnlyPublish = false) {
        static $commentList = array();
        $commentId = intval($commentId);
        if (!$commentId) {
            $this->setError('参数缺失');
            return false;
        }
        if (!isset($commentList[$commentId])) {
            $map = array('comment_id' => $commentId);

            $commentList[$commentId] = $this->commentModel->where($map)->find();
            if ($commentList[$commentId]) {
                $extend = M('ApplicationCommentData_' . $commentList[$commentId]['side_table_id'])->where($map)->find();
                if (!empty($extend)) {
                    $commentList[$commentId] = array_merge($extend, $commentList[$commentId]);
                }
            }
        }

        $commentInfo = $commentList[$commentId];
        if (is_array($commentInfo)) {
            // $isOnlyPublish只获取审核通过的评论
            if ($isOnlyPublish && !$commentInfo['status']) {
                return NULL;
            } else {
                return $commentInfo;
            }
        }
        return NULL;
    }

    /**
     * 根据程序id获取评分
     * @param type $applicationId
     * @return boolean
     */
    public function getScoreByApplicationId($applicationId) {
        $applicationId = intval($applicationId);
        if (!$applicationId) {
            $this->setError('参数缺失');
            return false;
        }
        $map = array('application_id' => $applicationId, 'status' => 1);
        $score = $this->commentModel->where($map)->avg('score');
        return ceil($score);
    }

    /**
     * 根据应用ID获取评论列表
     * @param type $applicationId               应用ID 
     * @param type $pageIndex                   页码
     * @param type $listRow                     每页数据条数
     * @param type $isOnlyPublish               是否只获取通过审核的评论
     * @return type
     */
    public function getCommentListByApplicationId($applicationId, $pageIndex = 1, $listRow = 10, $isOnlyPublish = false) {
        $applicationId = intval($applicationId);
        if (!$applicationId) {
            $this->setError('参数缺失');
            return false;
        }
        $map = array(
            'application_id' => $applicationId
        );
        if ($isOnlyPublish) {
            $map['status'] = 1;
        }
        $count = $this->commentModel->field('comment_id')->where($map)->count();
        $page = page($count, $listRow, $pageIndex);

        // 拉取评论列表
        $commentIdList = $this->commentModel->field('comment_id')->where($map)->order('comment_id DESC')->limit($page->firstRow, $page->listRows)->select();
        $commentList = array();
        foreach ($commentIdList as $commentInfo) {
            $commentInfo = $this->getCommentById($commentInfo['comment_id'], $isOnlyPublish);
            if ($commentInfo) {
                $commentList[] = $this->_formatNormalComment($commentInfo);
            }
        }
        $returnData = array(
            'commentList' => $commentList,
            'page' => $page,
        );
        return $returnData;
    }

    /**
     * 根据程序id获取评论数
     * @param type $applicationId
     */
    public function getCommentCountByApplicationId($applicationId, $isOnlyPublish = true) {
        $applicationId = intval($applicationId);
        if (!$applicationId) {
            $this->setError('参数缺失');
            return false;
        }
        $map = array(
            'application_id' => $applicationId
        );
        if ($isOnlyPublish) {
            $map['status'] = 1;
        }
        return $this->commentModel->field('comment_id')->where($map)->count();
    }

    /**
     * 获取评分最高的程序列表
     * @param type $applicationType             程序类型
     * @param type $pageIndex                   页码    
     * @param type $listRow                     每页数据条数
     * @return type
     */
    public function getScoreTopApplicationList($applicationType, $categoryId = null, $pageIndex = 1, $listRow = 10) {
        $pageIndex = max(1, intval($pageIndex));
        $listRow = intval($listRow);

        if (!$applicationType) {
            $this->setError('参数缺失');
            return false;
        }
        $applicationList = array();
        $map = array(
            'status' => 1,
            'flag' => $applicationType,
        );
        if ($categoryId) {
            $map['category_id'] = $categoryId;
        }
        $count = $this->commentModel->group('application_id')->where($map)->count();
        $page = page($count, $listRow, $pageIndex);
        if ($count) {
            $applicationList = $this->commentModel->group('application_id')->field('application_id,avg(score) as score')->where($map)->limit($page->firstRow, $page->listRows)->order('score DESC')->select();
        }
        return array('applicationList' => $applicationList, 'page' => $page);
    }

    /**
     * 格式化一般程度的评论
     * @param type $originComment
     * @return type
     */
    private function _formatNormalComment($originComment) {
        if ($originComment['user_id']) {
            $userName = getUserNameById($originComment['user_id']);
        } else {
            $userName = hideTheLastOneIp($originComment['author_ip']);
        }
        return array(
            'comment_id' => $originComment['comment_id'],
            'user_id' => $originComment['user_id'],
            'user_name' => $userName,
            'content' => $originComment['content'],
            'date' => date('Y-m-d H:i:s', $originComment['date']),
            'score' => intval($originComment['score']),
        );
    }

    /**
     * 检测评论参数/字段是否正确
     * @param type $content         评论内容
     * @param type $data            评论字段
     * @param type $userId          用户ID
     * @param type $pid             评论父ID
     * @return type                 成功返回处理后的$content，失败返回false
     */
    private function _checkArg($content, $data = array(), $userId = 0, $pid = 0) {
        //是否开启评论
        $setting = $this->setting;
        if (!$setting['application_comment_status']) {
            $this->setError('站点评论被关闭');
            return false;
        }

        // 评分判断
        if (intval($data['score']) <= 0) {
            $this->setError('请至少评1分');
            return FALSE;
        }

        //是否允许游客评论
        if (!$this->isInAdmin && !$setting['application_comment_is_allow_guest'] && $userId == 0) {
            $this->setError('请先登录');
            return false;
        }

        //评论时间检查
        $applicationCommentInterval = intval($setting['application_comment_interval']);
        if (!$this->isInAdmin && $applicationCommentInterval > 0 && time() - session('comment_time') < $applicationCommentInterval) {
            $this->setError('两次评论的间隔不能少于 ' . $applicationCommentInterval . ' 秒');
            return false;
        }

        //评论最大字节数
        if (empty($content)) {
            $this->setError('评论内容不能为空');
            return false;
        }

        $applicationCommentMaxLength = intval($setting['application_comment_max_length']);
        if ($applicationCommentMaxLength > 0 && strlen($content) > $applicationCommentMaxLength) {
            $this->setError('评论超出最大长度');
            return false;
        }

        //评论父对象是否存在
        if (!$this->isInAdmin && $pid && !$this->getCommentById($pid, true)) {
            $this->setError('回复的评论不存在');
            return false;
        }
        //过滤评论内容
        $content = BService('Filter')->filterContent($content);
        if (!$content) {
            $this->setError(BService('Filter')->getError());
            return false;
        }
        return $content;
    }

}

?>
