<?php

/**
 * 应用service
 * @author     bant
 * @copyright  (C) 2011-2013 BPICMS
 * @license    http://www.chinabpi.com/license/
 * @version    1.0
 * @createTime 2013-4-12
 * @updateTime 2013-4-12
 */
class ApplicationService extends BaseService {

    const ACTION_TYPE_GET_CONTENT_LIST_BY_MODEL_ID = 'getContentListByModelId'; //根据模型id获取内容列表
    const ACTION_TYPE_GET_CONTENT_LIST_BY_CATEGORY_ID = 'getContentListByCategoryId'; //根据栏目id获取内容列表

    protected $categoryId, $categoryCache, $modelId, $modelCache, $urlService;
    protected $fieldList;
    protected $applicationRelationModel; //关联内容模型
    protected $hitModel; //统计点击模型
    protected $isInAdmin; //是否在后台
    protected $htmlService; //htmlService
    protected $moduleId; //模块id

    public function __construct() {
        parent::__construct();
        $this->categoryCache = FData('Category');
        $this->modelCache = D('Model')->getModelByType(C('MODEL_TYPE.APPLICATION'), 1);
        $this->urlService = BService('Url');
        $this->htmlService = BService('Html');
        $this->isInAdmin = defined('IN_ADMIN') && IN_ADMIN;
        $this->applicationRelationModel = new AutoRelationModel('Application');
        $this->moduleId = C('ACCESS_TYPE.APPLICATION');
    }

    /**
     * 根据ID获取内容
     * @param type $contentId               文章ID
     * @param type $categoryId              模型ID，默认值为NULL，不为NULL时将检查category_id是否一致
     * @param type $modelId                 模型ID，默认值为NULL，不为NULL时将检查model_id是否一致
     * @param type $isGetRelationData       是否获取副表数据
     * @param type $isOnlyPublish           是否只获取成功发表的内容
     * @return type                         返回文章内容
     */
    public function getApplicationById($contentId, $categoryId = NULL, $modelId = NULL, $isGetRelationData = true, $isOnlyPublish = false) {
        if (empty($contentId)) {
            $this->setError('应用ID为空');
            return FALSE;
        }

        $map = array('id' => intval($contentId));
        !is_null($categoryId) && $map['category_id'] = intval($categoryId);
        !is_null($modelId) && $map['model_id'] = intval($modelId);
        $isOnlyPublish && $map['status'] = C('CONTENT_CHECK_STATUS.PASS');

        return $this->applicationRelationModel->relation($isGetRelationData)->where($map)->find();
    }

    /**
     * 静态根据ID获取内容
     * @staticvar array $contentList
     * @param type $contentId
     * @return type
     */
    public function staticGetContentById($contentId) {
        static $contentList = array();
        $contentId = intval($contentId);
        if (!isset($contentList[$contentId])) {
            $contentList[$contentId] = $this->getApplicationById($contentId);
        }
        return $contentList[$contentId];
    }

    /**
     * 获取内容数据列表
     * @param type $categoryId              栏目id
     * @param type $modelId                 模型id
     * @param type $pageIndex               页码
     * @param type $listRow                 每页返回数据条数
     * @param type $order                   排序
     * @param type $cacheTime               缓存时间
     * @return boolean
     */
    public function getContentList($categoryId = NULL, $modelId = NULL, $pageIndex = 1, $listRow = 20, $order = 'id DESC', $cacheTime = 3600) {
        $where = array('status' => C('CONTENT_CHECK_STATUS.PASS'));
        //TODO:严格这个构造id唯一性
        $cacheName = 'ApplicationList' . 'C' . $categoryId . 'M' . $modelId . 'P' . $pageIndex . 'L' . $listRow . 'O' . $order;
        if ($cacheTime && $return = S($cacheName)) {
            return $return;
        }
        //指定栏目id
        if ($categoryId) {
            $categoryId = intval($categoryId);
            $category = $this->categoryCache[$categoryId];
            if (!$category) {
                $this->setError('栏目不存在');
                return false;
            }
            //查询是否有子栏目
            if (empty($category['child_id_array'])) {//没有子栏目
                $where['category_id'] = $categoryId;
            } else {
                $where['category_id'] = array('IN', $category['child_id_array']);
            }
        }
        //指定模型id
        if ($modelId) {
            $where['model_id'] = $modelId;
        }
        //主表操作开始
        $count = $this->applicationRelationModel->relation(true)->where($where)->order($order)->count();
        $contentList = array();
        if ($count) {
            $page = page($count, $listRow, $pageIndex);
            $contentList = $this->applicationRelationModel->relation(true)->where($where)->order($order)->limit($page->firstRow . ',' . $page->listRows)->select();
        }
        $data = array(
            'contentList' => $contentList,
            'page' => $page,
        );
        //TODO:日志记录
        if ($cacheTime) {
            S($cacheName, $data, $cacheTime);
        }

        return $data;
    }

    /**
     * 根据栏目id获取内容列表(有缓存)
     * @param type $categoryId
     * @param type $modelId
     * @param type $pageIndex
     * @param type $listRow
     * @param type $order
     * @param type $cacheTime
     * @return boolean
     */
    public function getContentListByCategoryId($categoryId, $modelId = NULL, $pageIndex = 1, $listRow = 20, $order = 'id DESC', $cacheTime = 3600) {
        return $this->getContentList($categoryId, $modelId, $pageIndex, $listRow, $order, $cacheTime);
    }

    /**
     * 获取内容投稿/编辑页面html表格
     * @param type $categoryId
     * @param type $modelId
     * @param type $purviewAction
     * @param type $data
     * @return boolean
     */
    public function getApplicationForm($categoryId, $modelId, $purviewAction = '', $data = array()) {
        if (!$this->_checkCategoryIdAndModelId($categoryId, $modelId)) {
            return FALSE;
        }
        $data['model_id'] = $modelId;
        $data['category_id'] = $categoryId;

        //判断权限
        if ($this->isInAdmin) {//前后台区别判断
            //后台判断
            $isSystemAdmin = BaseAction::getVar('user_id') === BaseAction::getVar('super_role_id');
            if ($isSystemAdmin) {//超级管理员直接通过
                $pruview = true;
            } else {
                $pruview = D('CategoryPurview')->where(array('category_id' => $categoryId, 'is_admin' => intval($this->isInAdmin), 'role_id' => BaseAction::getVar('user/role_id'), 'action' => $purviewAction))->find();
            }
        } else {
            //前台判断
            $pruview = D('CategoryPurview')->where(array('category_id' => $categoryId, 'is_admin' => intval($this->isInAdmin), 'role_id' => BaseAction::getVar('user/group_id'), 'action' => $purviewAction))->find();
        }
        if (!$pruview) {
            $this->setError('没有此栏目投稿权限');
            return false;
        }

        //导入生成内容表单类
        import('contentForm', RUNTIME_PATH);
        $contentFormClass = new ContentForm($modelId);
        $formModuleList = $contentFormClass->get($data);


        //获得格式化的表单组件列表
        if (!empty($formModuleList)) {
            foreach ($formModuleList as $field => $value) {
                //万能附属字段则跳过
                if ($value['is_omnipotent']) {
                    continue;
                }
                if ($value['field_type'] == 'omnipotent') {//万能字段
                    foreach ($formModuleList['base'] as $_fm => $_fm_value) {
                        if ($_fm_value['is_omnipotent']) {
                            $formModuleList[$field]['form_module'] = str_replace('{' . $_fm . '}', $_fm_value['form_module'], $value['form_module']);
                        }
                    }
                    foreach ($formModuleList['extend'] as $_fm => $_fm_value) {
                        if ($_fm_value['is_omnipotent']) {
                            $formModuleList[$field]['form_module'] = str_replace('{' . $_fm . '}', $_fm_value['form_module'], $value['form_module']);
                        }
                    }
                }
            }
        }
        return $formModuleList;
    }

    /**
     * 添加内容
     * @param type $data
     * @param type $isImport 是否直接入库处理，跳过提示报错
     * @return type
     */
    public function add($data, $isImport = false) {
        return $this->_saveApplication(C('ACTION_TYPE.ADD'), $data, $isImport);
    }

    /**
     * 文章删除
     * @param array $ids            文章ID，支持单个ID，支持ID数组
     * @param type $map             辅助筛选条件，防止错误删除，可以为空
     * @return boolean              返回是否删除成功
     */
    public function delete($ids, $map = array()) {
        //TODO:删除内容后的后置操作没有完成
        if (empty($ids)) {
            $this->setError('应用ID为空');
            return FALSE;
        }
        // 单个ID转换成数组
        if (is_array($ids)) {
            $ids = array_unique(array_values($ids));
        } else {
            $ids = array($ids);
        }

        // 实例化关系模型
        $count = 0;
        foreach ($ids as $id) {
            // 获取文章内容
            $map['id'] = $id;
            $data = $this->applicationRelationModel->relation(true)->where($map)->find();
            if (empty($data)) {
                continue;
            }
            // 删除静态页面
            $this->htmlService->deleteHtml($data['category_id'], $data['id'], $data['post_time'], $data['alias']);
            // 删除内容
            $this->applicationRelationModel->relation(true)->where($map)->delete() && $count++;
            //调用contentDelete类执行操作
            import('contentDelete', RUNTIME_PATH);
            $contentDeleteClass = new ContentDelete($data['model_id']);
            $contentDeleteClass->delete($data);

            // 删除附件
            BService('Attachment')->apiDelete($this->moduleId, $id, NULL, NULL, NULL);
            // 删除点击量信息
            BService('Hit')->delete($this->moduleId, $id);
            // 删除评论
            BService('ApplicationComment', 'Market')->deleteByApplicationId($id);
            // 删除推荐位
            BService('Position')->deleteData($data['model_id'], $id);
            // TODO : 删除对应的会员投稿记录信息
            // TODO : 删除全站搜索数据
            $this->_searchApi($id, $data, 'delete');
            // TODO:删除收藏
            BService('Favorite', 'Favorite')->deleteFavorite(NULL, $this->moduleId, $id);
            // 删除所有引用文章
            $this->applicationRelationModel->where(array('pid' => $id))->delete();
        }

        if (!$count) {
            $this->setError('没有删除任何数据');
            return FALSE;
        } else {
            return TRUE;
        }
    }

    /**
     * 编辑内容
     * @param type $data post数据
     * @return type
     */
    public function edit($data) {
        return $this->_saveApplication(C('ACTION_TYPE.EDIT'), $data);
    }

    /**
     * 推送到其他栏目
     * @param type $modelId         模型ID
     * @param type $categoryId      栏目ID
     * @param type $pushData        推送数据
     */
    public function push($modelId, $categoryId, $pushData = array()) {
        if (!is_array($pushData) || empty($pushData)) {
            $this->setError('推送数据为空');
            return FALSE;
        }

        $modelId = intval($modelId);
        $categoryId = intval($categoryId);

        // 特殊处理某些字段
        $data = array_merge($pushData, array(
            'pid' => $pushData['id'],
            'category_id' => $categoryId,
            'model_id' => $modelId,
            'post_time' => time(),
            'update_time' => time(),
            'is_admin_add' => 1,
        ));
        unset($data['id']);

        $applicationRelationModel = new AutoRelationModel('Application');

        $result = $applicationRelationModel->field('id')->where(array('id|pid' => $data['pid'], 'category_id' => $categoryId, 'model_id' => $modelId))->find();
        if ($result) {
            $result = $result['id'];
        } else {
            $result = $applicationRelationModel->relation(TRUE)->add($data);
        }
        return $result;
    }

    /**
     * 改变应用状态
     * @param type $applicationIdList   应用ID列表
     * @param type $status              应用状态
     */
    public function changeStatus($applicationIdList, $status) {
        if (!is_array($applicationIdList)) {
            $applicationIdList = array(intval($applicationIdList));
        }
        $status = intval($status);

        return $this->applicationRelationModel->where(array('id' => array('IN', $applicationIdList)))->save(array('status' => $status));
    }

    /**
     * 检查应用升级
     * @param array $data   检查升级的应用数据
     * @return array
     */
    public function applicationsCheckUpdate($data) {
        if (empty($data)) {
            $this->setError('传入数据为空');
            return FALSE;
        }

        // 提取包名列表
        $packageNameList = array();
        foreach ($data as $applicationInfo) {
            $packageNameList[] = $applicationInfo['package_name'];
        }

        // 找出应用信息
        $applicationList = $this->applicationRelationModel->relation(TRUE)->where(array('package_name' => array('IN', $packageNameList), 'status' => C('CONTENT_CHECK_STATUS.PASS')))->select();

        // 开始检查更新
        $applicationCheckUpdateList = setArrayIndexByKey($data, 'package_name');
        $result = array();
        foreach ($applicationList as $applicationInfo) {
            if (intval($applicationCheckUpdateList[$applicationInfo['package_name']]['version_code']) < intval($applicationInfo['version_code'])) {
                $result[] = BService('MarketView', 'Index')->formatSimpleApplicaition($applicationInfo);
            }
        }

        return $result;
    }

    /**
     * 获取处理过的时间格式
     * @param type $targetTime
     * @return type
     */
    private function _getHandleTime($targetTime) {
        if ($targetTime) {
            $time = is_numeric($targetTime) ? $targetTime : strtotime($targetTime);
        } else {
            $time = time();
        }
        return $time;
    }

    /**
     * 更新搜索数据
     */
    private function _searchApi() {
        //TODO:
    }

    /**
     * 同步发布到其他栏目
     */
    private function _post_to_other_category() {
        //TODO:
    }

    /**
     * 生成上一篇下一篇内容
     */
    private function _generate_previous_next_content() {
        //TODO:
    }

    /**
     * 保存内容
     * @param type $actionType
     * @param type $postData
     * @param type $extParam
     * @return boolean
     */
    private function _saveApplication($actionType, $postData, $extParam = NULL) {
        if (!$this->_checkCategoryIdAndModelId($postData['category_id'], $postData['model_id'])) {
            return FALSE;
        }

        $data = $postData['info'];
        $data['id'] = $postData['id'];
        $data['model_id'] = $postData['model_id'];
        $data['category_id'] = $postData['category_id'];

        $isImport = FALSE;
        if ($actionType == C('ACTION_TYPE.ADD')) {//add
            $isActionAdd = true;
            $isImport = $extParam;
            $id = $data['id'] = $postData['id'] = 0;
        } else {//edit
            $isActionAdd = false;
            $id = $data['id'] = intval($postData['id']);
        }

        //验证用户
        if (!$isActionAdd) {
            if ($id) {
                $originData = $this->getApplicationById($id, $postData['category_id'], $postData['model_id'], false);
                if (!$originData) {
                    $this->setError('内容不存在');
                    return false;
                }

                if (!$this->isInAdmin) {
                    if ($originData['is_admin_add'] == 1 || $originData['user_id'] != BaseAction::getVar('user_id')) {
                        $this->setError('此内容无权限操作');
                        return false;
                    }
                }
            } else {
                $this->setError('参数缺失');
                return false;
            }
        }

        $this->categoryId = intval($postData['category_id']);
        $this->modelId = intval($postData['model_id']);

        //导入字段处理类
        import('contentInput', RUNTIME_PATH);
        import('contentUpdate', RUNTIME_PATH);
        $contentInputClass = new ContentInput($this->modelId);
        if (!$contentInputClass) {//实例化失败
            $this->setError('输入处理类实例化失败');
            return false;
        }
        //获得 进行输入处理的数据
        $inputInfo = $contentInputClass->get($data, $isImport);
        if (empty($inputInfo)) {
            $this->setError($contentInputClass->getError());
            return false;
        }

        //主表字段内容
        $systemInfo = is_array($inputInfo['system']) ? $inputInfo['system'] : array();
        $systemInfo['id'] = $id;
        $systemInfo['model_id'] = $this->modelId;
        $systemInfo['category_id'] = $this->categoryId;
        //副表内容
        $modelInfo = is_array($inputInfo['model']) ? $inputInfo['model'] : array();
        //栏目数据
        $category = $this->categoryCache[$this->categoryId];
        //栏目配置
        $categorySetting = $category['setting'];

        //前台判断
        if (!$this->isInAdmin) {
            //前台投稿，根据栏目配置和用户配置
            $memberGroupCache = FData('MemberGroup');
            $groupId = BaseAction::getVar('user/group_id');
            //如果会员组设置中设置，投稿不需要审核，直接无视栏目设置
            if ($memberGroupCache[$groupId]['is_allow_post_no_verify']) {
                $systemInfo['status'] = C('CONTENT_CHECK_STATUS.PASS');
            } else {
                if ($categorySetting['member_check']) {
                    $systemInfo['status'] = C('CONTENT_CHECK_STATUS.NORMAL');
                } else {
                    $systemInfo['status'] = C('CONTENT_CHECK_STATUS.PASS');
                }
            }
        }

        if ($isActionAdd) {
            //处理 真实发表时间（取当前时间）
            $systemInfo['post_time'] = time();
            //处理 添加用户名（暂不支持自定义投递人名称）
            //采集时候注释掉
            //$systemInfo['user_id'] = BaseAction::getVar('user_id');
            $systemInfo['user_id'] = '1';

            //处理 是否后台发布
            $systemInfo['is_admin_add'] = intval($this->isInAdmin);
        } else {
            //真实发表时间，不允许修改
            unset($systemInfo['post_time']);
        }

        //处理 更新时间
        $systemInfo['update_time'] = self::_getHandleTime($systemInfo['update_time']);

        $content = stripslashes($modelInfo['content']); //内容字段
        //处理 自动摘要, 设置自动提取,且摘要为空，有内容字段才执行
        if ($data['is_auto_description'] && empty($modelInfo['description']) && $content) {
            $descriptionLength = intval($data['description_length']);
            //TODO:过滤内容
            $contentAfterFilter = str_replace(array("\r\n", "\t", '[page]', '[/page]', '&ldquo;', '&rdquo;', '&nbsp;'), '', strip_tags($content));
            $systemInfo['description'] = cutString($contentAfterFilter, 0, $descriptionLength);
        }

        //处理 自动提取缩略图,TODO:这里保存的是URL，需要进一步处理
//        if ($data['is_auto_thumb'] && empty($modelInfo['thumb']) && $content) {
//            $autoThumbIndex = intval($data['auto_thumb_index']) - 1;
//            $matches = array();
//            if (preg_match_all("/(src)=([\"|']?)([^ \"'>]+\.(gif|jpg|jpeg|bmp|png))\\2/i", $content, $matches)) {
//                $modelInfo['thumb'] = $matches[3][$autoThumbIndex];
//            }
//        }
        //TODO:插入内容索引表
        //合并令牌，TODO:没什么用，只能对主表进行create，如果我只更新附表内容，此处毫无作用
        $systemInfo[C('TOKEN_NAME')] = $data[C('TOKEN_NAME')];

        //合并数据并保存内容(前置，进行，后置操作)
        if ($isActionAdd) {
            //保存内容前置操作-合并数据
            $mergeMainSideData = $this->_getMergeMainSideData($systemInfo, $modelInfo);

            //保存内容
            $id = $this->applicationRelationModel->relation(true)->add($mergeMainSideData);
            if (empty($id)) {
                $this->setError('信息添加失败');
            }
            $systemInfo['id'] = $id;

            // 处理转向连接
//            $data['is_link'] && $systemInfo['url'] = $postData['is_link_url'];
//            if (empty($systemInfo['url'])) {
//                $systemInfo['url'] = $this->urlService->getContentUrl($this->categoryId, $id, $systemInfo['post_time'], 0, $systemInfo['alias']);
//            }
            //更新url 
            $this->applicationRelationModel->where(array('id' => $id))->save(array('url' => $systemInfo['url']));

            //处理点击统计
            //TODO:在适当的地方调用hitservice，生成对应hit数据
//            $this->_handleHit($systemInfo, $id);
            //处理同时发布到其他栏目
            $this->_handlePostOtherCategory($data);
        } else {
            // 处理转向连接
//            $data['is_link'] && $systemInfo['url'] = $postData['is_link_url'];
//            if (empty($systemInfo['url'])) {
//                $systemInfo['url'] = $this->urlService->getContentUrl($this->categoryId, $id, $systemInfo['post_time'], 0, $systemInfo['alias']);
//            }
            //合并数据
            $mergeMainSideData = $this->_getMergeMainSideData($systemInfo, $modelInfo);
            //保存内容
            $status = $this->applicationRelationModel->relation(true)->where(array('id' => $id))->save($mergeMainSideData);
            if ($status === false) {
                $this->setError($this->applicationRelationModel->getError());
            }
        }

        //处理 全站搜索
        $contentStatus = intval($systemInfo['status']);
        if ($contentStatus === C('CONTENT_CHECK_STATUS.PASS')) {
            //TODO:完成此方法
            $this->_searchApi($id, $inputInfo);
        }

        //处理 更新操作（生成的数据中的字段对应更新操作）
        $contentData = array_merge($systemInfo, $modelInfo);
        $contentUpdateClass = new ContentUpdate($this->modelId);
        $updateData = $contentUpdateClass->update($contentData);
        if (!empty($updateData)) {
            $updateData = $this->_getMergeMainSideData($updateData['system'], $updateData['model']);
            // 加上model_id，用于防止数据保存失败和自动检测副表模型
            $updateData['model_id'] = $contentData['model_id'];
            $this->applicationRelationModel->relation(true)->where(array('id' => $id))->save($updateData);
        }

        //处理 生成相关
        $generateListIndex = 0;
        if ($this->isInAdmin) {
            //是否生成内容页
            if (intval($categorySetting['generate_html']) === 1) {
                //生成静态
                if ($categorySetting['content_is_set_html'] && $contentStatus === C('CONTENT_CHECK_STATUS.PASS')) {
                    $this->htmlService->buildContent($contentData);
                }
            }
            //如果未审核，删除已经生成的静态内容页(编辑保存时，判断)
            if (!$isActionAdd && $categorySetting['content_is_set_html'] && !$data['is_link'] && $contentStatus === C('CONTENT_CHECK_STATUS.NORMAL')) {
                $this->htmlService->deleteHtml($systemInfo['category_id'], $id, $systemInfo['post_time'], $systemInfo['alias']);
            }
            //是否生成列表
            if (intval($categorySetting['generate_list']) > 0) {
                $generateListIndex = intval($categorySetting['generate_list']);
            }
        } else {
            //投稿内容页生成，直接审核通过的生成内容页
            if ($contentStatus === C('CONTENT_CHECK_STATUS.PASS')) {
                //生成静态
                if ($categorySetting['content_is_set_html']) {
                    $this->htmlService->buildContent($contentData);
                }
            } elseif (!$isActionAdd && $categorySetting['content_is_set_html'] && !$data['is_link']) {//编辑发现没有审核通过，则删除静态
                $this->htmlService->deleteHtml($systemInfo['category_id'], $id, $systemInfo['post_time'], $systemInfo['alias']);
            }
            //是否生成列表
            if (intval($categorySetting['member_generate_list']) > 0) {
                $generateListIndex = intval($categorySetting['member_generate_list']);
            }
        }
        //列表生成
        $generateType = C('GENERATE_TYPE');
        // 生成首页
        if ($generateListIndex & $generateType['GENERATE_INDEX'][0]) {
            $this->htmlService->buildIndex();
        }

        // 生成当前栏目
        if ($generateListIndex & $generateType['GENERATE_CATEGORY'][0]) {
            $this->htmlService->buildRelationCategory($mergeMainSideData['category_id']);
        }

        // 生成父栏目
        if ($generateListIndex & $generateType['GENERATE_PARENT_CATEGORY'][0]) {
            if (0 !== ($pid = intval($category['pid']))) {
                $this->htmlService->buildRelationCategory($pid);
            }
        }

        //处理 生成上一篇，下一篇(编辑时候，直接执行)
        if (!$isActionAdd || $contentStatus === C('CONTENT_CHECK_STATUS.PASS')) {
            $this->_generate_previous_next_content();
        }

        return $id;
    }

    /**
     * 获取 合并主副表数据
     * @param type $systemInfo
     * @param type $modelInfo
     * @return type
     */
    private function _getMergeMainSideData($systemInfo, $modelInfo) {
        //Model类验证数据
        $resultData = $this->applicationRelationModel->create($systemInfo);
        if (empty($resultData)) {
            $this->setError($this->applicationRelationModel->getError());
        }
        //合并副表数据,PS:验证$resultData，防止为false时,sideData合并之后为NULL，导致sql报错
        $resultData[C('SIDE_TABLE_DATA_FIELD')] = (array) $modelInfo;
        return $resultData;
    }

    /**
     * 处理 点击统计
     * @param type $systemInfo
     * @param type $id
     */
//    private function _handleHit($systemInfo, $id) {
    //处理 点击统计
//        $this->hitModel = D('Hit');
//        $hitId = makeKey(C('KEY_TYPE.HIT'), C('HIT_TYPE_PREFIX.CATEGORY'), $systemInfo['category_id'], $id);
//        $this->hitModel->add(array('hit_id' => $hitId, 'category_id' => $systemInfo['category_id'], 'update_time' => time()));
//    }

    /**
     * 处理 同时发送到其他栏目
     * @param type $data
     */
    private function _handlePostOtherCategory($data) {
        //处理 发布到其他栏目
        if ($this->isInAdmin) {
            if (is_array($data['other_category_id_array'])) {
                foreach (array_keys($data['other_category_id_array']) as $otherCategoryId) {
                    if ($this->categoryId == $otherCategoryId) {
                        continue;
                    }
                    $otherCategoryIdList[] = $otherCategoryId;
                }
                //去重复
                $otherCategoryIdList = array_unique($otherCategoryIdList);
                //TODO:完成此方法
                $this->_post_to_other_category();
            }
        }
    }

    /**
     * 检查栏目ID与模型ID
     * @param type $categoryId      栏目ID
     * @param type $modelId         模型ID
     * @return boolean              有关联返回TRUE，无关联返回FALSE
     */
    private function _checkCategoryIdAndModelId($categoryId, $modelId) {
        $categoryId = intval($categoryId);
        $modelId = intval($modelId);

        $category = $this->categoryCache[$categoryId];
        if (is_array($category)) {
            if (array_key_exists($modelId, $category['model'])) {
                if (array_key_exists($modelId, $this->modelCache)) {
                    return TRUE;
                }
            }
        }

        $this->setError('栏目ID与模型ID不匹配');
        return FALSE;
    }

}

?>
